/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.1.2.7 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/agp_backend.h>
#include "agp.h"
#include "global.h"
#include "igd_gart.h"
#include "intelpci.h"
#include "igd_abs.h"

static gart_dispatch_t gart_pci_device_table[] = {
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_810, PCI_DEVICE_ID_810,
	 "810", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_810DC, PCI_DEVICE_ID_810DC,
	 "810DC", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_810E, PCI_DEVICE_ID_810E,
	 "810E", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_815, PCI_DEVICE_ID_815,
	 "815", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_830M, PCI_DEVICE_ID_830M,
	 "830M", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_835, PCI_DEVICE_ID_835,
	 "835", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_845G, PCI_DEVICE_ID_845G,
	 "845G", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_855, PCI_DEVICE_ID_855,
	 "855", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_865G, PCI_DEVICE_ID_865G,
	 "865G", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_915GD, PCI_DEVICE_ID_915GD,
	 "915GD", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_910GL, PCI_DEVICE_ID_910GL,
	 "910GL", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_915AL, PCI_DEVICE_ID_915AL,
	 "915AL", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_945G, PCI_DEVICE_ID_945G,
	 "945G", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_945GM, PCI_DEVICE_ID_945GM,
	 "945GM", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_945GME, PCI_DEVICE_ID_945GME,
	 "945GME", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_Q35, PCI_DEVICE_ID_Q35,
	 "Q33/Q35", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_965G, PCI_DEVICE_ID_965G,
	 "965G", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_946GZ, PCI_DEVICE_ID_946GZ,
	 "946GZ", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_G965, PCI_DEVICE_ID_G965,
	 "G965", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_Q965, PCI_DEVICE_ID_Q965,
	 "Q965", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_GM965, PCI_DEVICE_ID_GM965,
	 "GM965", 0, 0, 0,
	},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BRIDGE_GME965, PCI_DEVICE_ID_GME965,
	 "GME965", 0, 0, 0,
	},
};

/* PCI device id that supported by IEGD */
struct pci_device_id iegd_pci_table[] = {
	ID(PCI_DEVICE_ID_BRIDGE_810),
	ID(PCI_DEVICE_ID_BRIDGE_810DC),
	ID(PCI_DEVICE_ID_BRIDGE_810E),
	ID(PCI_DEVICE_ID_BRIDGE_815),
	ID(PCI_DEVICE_ID_BRIDGE_830M),
	ID(PCI_DEVICE_ID_BRIDGE_845G),
	ID(PCI_DEVICE_ID_BRIDGE_855),
	ID(PCI_DEVICE_ID_BRIDGE_865G),
	ID(PCI_DEVICE_ID_BRIDGE_915GD),
	ID(PCI_DEVICE_ID_BRIDGE_915AL),
	ID(PCI_DEVICE_ID_BRIDGE_945G),
	ID(PCI_DEVICE_ID_BRIDGE_945GM),
	ID(PCI_DEVICE_ID_BRIDGE_945GME),
	ID(PCI_DEVICE_ID_BRIDGE_965G),
	ID(PCI_DEVICE_ID_BRIDGE_946GZ),
	ID(PCI_DEVICE_ID_BRIDGE_G965),
	ID(PCI_DEVICE_ID_BRIDGE_Q965),
	ID(PCI_DEVICE_ID_BRIDGE_GM965),
	ID(PCI_DEVICE_ID_BRIDGE_GME965),
	ID(PCI_DEVICE_ID_BRIDGE_Q35),
	{ }
};

MODULE_DEVICE_TABLE(pci, iegd_pci_table);


/* This function get called by PCI core when one of the chipset
 * above detected */
static int __devinit iegd_intel_probe(
	struct pci_dev *pdev,
	const struct pci_device_id *ent)
{
	bridge_data_t *bridge_device;
	u8 cap_ptr = 0;
	struct resource *r;
	int ret;

	AGN_DEBUG("Enter");
	AGN_LOG("Initialize IEGD agpgart and drm");

	/* Make sure this probing is called because of prefered
	 * chipsets. This is because to make sure we initialize
	 * chipset that belong to deregister gart module */
	if(!gart_id->bridge_pdev ||
		(gart_id->bridge_pdev->device != pdev->device)) {
		return -ENODEV;
	}

	cap_ptr = pci_find_capability(pdev, PCI_CAP_ID_AGP);

	/* Allocate memory for the bridge. This data structure then will be
	 * used by the agp backend and frontend */
	gart_id->bridge_info = agp_alloc_bridge();
	if(gart_id->bridge_info == NULL) {
		return -ENOMEM;
	}

	/* Check for the device and initialize private data */
	if(!iegd_find_device(gart_id->device_id)) {
		agp_put_bridge(gart_id->bridge_info);
		AGN_ERROR("Unsupported device: %x", gart_id->device_id);
		return -ENODEV;
	}

	bridge_device = gart_id->bridge_info;

	/* Dispatch the core function based on the chipset id */
	ret = bridge_driver_init(&bridge_device->driver,
			gart_id->device_id, driver_dispatch_list);

	if(ret) {
		agp_put_bridge(gart_id->bridge_info);
		AGN_ERROR("Device found but no function hook");
		return -ENODEV;
	}

	bridge_device->dev = pdev;
	bridge_device->capndx = cap_ptr;
	bridge_device->dev_private_data = &private_data;

	AGN_LOG("Intel %s chipset detected", gart_id->name);

	r = &pdev->resource[0];
    if (!r->start && r->end) {
		if (pci_assign_resource(pdev, 0)) {
			AGN_ERROR("could not assign resource 0");
			agp_put_bridge(gart_id->bridge_info);
			return -ENODEV;
		}
	}

	if(pci_enable_device(pdev)) {
		AGN_ERROR("Unable to enable PCI device");
		agp_put_bridge(gart_id->bridge_info);
		return -ENODEV;
	}

	if(cap_ptr) {
		pci_read_config_dword(pdev,
				bridge_device->capndx+PCI_AGP_STATUS,
				&bridge_device->mode);
	}

	pci_set_drvdata(pdev, bridge_device);
	AGN_DEBUG("Exit");
	return agp_add_bridge(bridge_device);
}

static int iegd_intel_remove(struct pci_dev *pdev)
{
	AGN_LOG("Exit from module");
	return 0;
}

static struct pci_driver iegd_pci_driver = {
	.name		= "iegd-intel",
	.id_table	= iegd_pci_table,
	.probe		= iegd_intel_probe,
	.remove		= __devexit_p(iegd_intel_remove),
	/*
	.suspend    = agp_intel_suspend,
	.resume		= agp_intel_resume,
	*/
};

struct pci_dev *iegd_probe_device()
{
	int i;
	struct pci_dev *dev;

	AGN_DEBUG("Enter");

	/* Probed for the supported devices */
	for(i=0 ; i<sizeof(gart_pci_device_table)/sizeof(gart_dispatch_t);
			i++) {
		dev = pci_get_device(PCI_VENDOR_ID_INTEL,
			(unsigned int)gart_pci_device_table[i].bridge_id, NULL);
		if(dev){
			gart_id = &gart_pci_device_table[i];
			AGN_DEBUG("Device found.. Exit");
			return dev;
		}
	}
	AGN_DEBUG("Device not found.. Exit");
	return NULL;
}

/**
 * This is the first routine been called by the init function.
 * This function will probe for devices that supported by IEGD.
 * Once it found the device, it will check whether driver for
 * this device exist. If it exist, get the pci_driver structure
 * for the existing driver and call the pci unregister fucntion
 * to deregister existing driver and register iegd agpgart
 */
static int iegd_agp_init(void)
{
	struct pci_driver *curr_driver;
	struct pci_dev *temp_pdev;

	AGN_DEBUG("Enter");

	/* Probe for the intel embedded graphic device chipset */
	temp_pdev = iegd_probe_device();

	if(!temp_pdev) {
		return -ENODEV;
	}

	gart_id->bridge_pdev = temp_pdev;
	curr_driver = pci_dev_driver(gart_id->bridge_pdev);

	if(curr_driver) {
		/* FIXME: Don't know whether we have to make separate copy of this
		 * structure */
		gart_id->old_gart  = curr_driver;

		/* deregister pci driver from pci core. This is needed since we
		 * don't want 2 agpgart reside in the kernel that respond to the
		 * same device id */
		AGN_LOG("Unregister agpgart name %s", curr_driver->name);
		pci_unregister_driver(curr_driver);
	}

	AGN_LOG("Registering iegd gart module");
	/* Register our own to pci core */
	AGN_DEBUG("Exit");
	return pci_register_driver(&iegd_pci_driver);

}

static void iegd_restore_device(void)
{
	int ret;

	AGN_DEBUG("Enter");

	/* Decrement the reference for this pci device */
	pci_dev_put(gart_id->bridge_pdev);

	if(gart_id->old_gart) {
		/* Register the original driver */
		ret = pci_register_driver(gart_id->old_gart);
	}

	AGN_DEBUG("Exit");

}

static int iegd_gart_init(void)
{
	int ret;

	AGN_DEBUG("Enter");

	/* Find bridge based on chipset supported by IEGD */
	ret = iegd_agp_init();
	if(AGP_RET(ret)) {
		AGN_LOG("Registering iegd drm module");
		/* Initialize DRM module by calling DRM init function */
		return DRM_INIT_MODULE();
	} else {
		/* Log the driver failed to register */
		AGN_LOG("Driver registration failed");
	}

	/* Return agp error if agp init failed */
	return ret;

	AGN_DEBUG("Exit");
}

static void iegd_gart_exit(void)
{
	/* Unregister DRM module */
	DRM_EXIT_MODULE();

	/* Remove our device from the kernel */
	pci_unregister_driver(&iegd_pci_driver);

	/* Restore back the old agp gart */
	iegd_restore_device();
}

MODULE_LICENSE("GPL and additional rights");

module_init(iegd_gart_init);
module_exit(iegd_gart_exit);
