/****************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ****************************************************************************

    Module Name:
    soft_ap.c

    Abstract:
    Access Point specific routines and MAC table maintenance routines

    Revision History:
    Who         When          What
    --------    ----------    ----------------------------------------------
    John Chang  08-04-2003    created for 11g soft-AP

 */

#include "rt_config.h"

char const *pEventText[EVENT_MAX_EVENT_TYPE] = {
	"restart access point",
	"successfully associated",
	"has disassociated",
	"has been aged-out and disassociated" ,
	"active countermeasures",
	"has disassociated with invalid PSK password"};


/*
	==========================================================================
	Description:
		Initialize AP specific data especially the NDIS packet pool that's
		used for wireless client bridging.
	==========================================================================
 */
NDIS_STATUS APInitialize(
	IN  PRTMP_ADAPTER   pAd)
{
	NDIS_STATUS     Status = NDIS_STATUS_SUCCESS;
	//UCHAR   GTK[TKIP_GTK_LENGTH];

	DBGPRINT(RT_DEBUG_TRACE, ("---> APInitialize\n"));
# if 0 // Dennis Lee
    NdisAllocatePacketPool(
        &Status,
        &pAd->FreeNdisPacketPoolHandle,
        MAX_NUM_OF_FREE_NDIS_PACKET,
        sizeof(PVOID) * 4);

    if (Status != NDIS_STATUS_SUCCESS)
        return Status;

    NdisAllocateBufferPool(
        &Status,
        &pAd->FreeNdisBufferPoolHandle,
        MAX_NUM_OF_FREE_NDIS_PACKET);

    if (Status != NDIS_STATUS_SUCCESS)
        return Status;

#endif
	// initialize MAC table and allocate spin lock
	NdisZeroMemory(&pAd->MacTab, sizeof(MAC_TABLE));
	NdisAllocateSpinLock(&pAd->MacTabLock);

	// Init TKIP Group-Key-related variables
//	APGenRandom(pAd,pAd->ApCfg.GMK);
//	APGenRandom(pAd,pAd->ApCfg.GNonce);
//	CountGTK(pAd->ApCfg.GMK, (UCHAR*)pAd->ApCfg.GNonce, pAd->CurrentAddress, GTK, TKIP_GTK_LENGTH);
//	NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key, GTK, 16);
//	NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic, &GTK[16], 8);
//	NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic, &GTK[24], 8);

	// Init Group key update timer, and countermeasures timer
#ifdef WIN_NDIS
	NdisMInitializeTimer(&pAd->ApCfg.REKEYTimer, pAd->AdapterHandle, GREKEYPeriodicExec, pAd);
	NdisMInitializeTimer(&pAd->ApCfg.CounterMeasureTimer, pAd->AdapterHandle, CMTimerExec, pAd);
	NdisMInitializeTimer(&pAd->ApCfg.ApQuickResponeForRateUpTimer, pAd->AdapterHandle, APQuickResponeForRateUpExec, pAd);
#else
	RTMPInitTimer(pAd, &pAd->ApCfg.REKEYTimer,GET_TIMER_FUNCTION(GREKEYPeriodicExec), pAd, TRUE);
	RTMPInitTimer(pAd, &pAd->ApCfg.CounterMeasureTimer, GET_TIMER_FUNCTION(CMTimerExec), pAd, FALSE);
    /* ~~sample, modify, waked up only in APMlmeDynamicTxRateSwitching(), 2006/10/13 */
	RTMPInitTimer(pAd, &pAd->ApCfg.ApQuickResponeForRateUpTimer, GET_TIMER_FUNCTION(APQuickResponeForRateUpExec), pAd, FALSE);
	RTMPInitTimer(pAd, &pAd->ApCfg.TBTTTimer, GET_TIMER_FUNCTION(MlmeTBTTExec), pAd, TRUE);
#endif
	pAd->ApCfg.ApQuickResponeForRateUpTimerRunning = FALSE;

	NdisZeroMemory(&pAd->WdsTab, sizeof(WDS_TABLE));
	NdisAllocateSpinLock(&pAd->WdsTabLock);

	DBGPRINT(RT_DEBUG_TRACE, ("<--- APInitialize\n"));
	return Status;
}

/*
	==========================================================================
	Description:
		Shutdown AP and free AP specific resources
	==========================================================================
 */
VOID APShutdown(
	IN PRTMP_ADAPTER pAd)
{
	DBGPRINT(RT_DEBUG_TRACE, ("---> APShutdown\n"));
	if (pAd->OpMode == OPMODE_AP)
	APStop(pAd);
	NdisFreeSpinLock(&pAd->MacTabLock);
	NdisFreeSpinLock(&pAd->WdsTabLock);
	DBGPRINT(RT_DEBUG_TRACE, ("<--- APShutdown\n"));
}

/*
	==========================================================================
	Description:
		Start AP service. If any vital AP parameter is changed, a STOP-START
		sequence is required to disassociate all STAs.

	IRQL = DISPATCH_LEVEL.(from SetInformationHandler)
	IRQL = PASSIVE_LEVEL. (from InitializeHandler)

	Note:
		Can't call NdisMIndicateStatus on this routine.

		RT61 is a serialized driver on Win2KXP and is a deserialized on Win9X
		Serialized callers of NdisMIndicateStatus must run at IRQL = DISPATCH_LEVEL.

	==========================================================================
 */
VOID APStartUp(
	IN PRTMP_ADAPTER pAd)
{
//	ULONG         temp_counter=97;
	UCHAR         GTK[TKIP_GTK_LENGTH];
	BOOLEAN       Cancelled;
	int i;

	DBGPRINT(RT_DEBUG_TRACE, ("===> APStartUp\n"));

	//
	// First cancel the Link down timer.
	//
	RTMPCancelTimer(&pAd->Mlme.LinkDownTimer, &Cancelled);

	//NICEnableInterrupt(pAd);
	if ((pAd->CommonCfg.SsidLen <= 0) || (pAd->CommonCfg.SsidLen > MAX_LEN_OF_SSID))
	{
		NdisMoveMemory(pAd->CommonCfg.Ssid, "My AP", 5);
		pAd->CommonCfg.SsidLen = 5;
	}

    /* ~~sample, bug fix for Tx Preamble setup, 2006/10/04 */
//	pAd->ApCfg.CapabilityInfo    = CAP_GENERATE(1, 0, (pAd->CommonCfg.WepStatus != Ndis802_11EncryptionDisabled), 1, pAd->CommonCfg.bUseShortSlotTime);
	pAd->ApCfg.CapabilityInfo    = CAP_GENERATE(1, 0, (pAd->CommonCfg.WepStatus != Ndis802_11EncryptionDisabled), (pAd->CommonCfg.TxPreamble == Rt802_11PreambleShort), pAd->CommonCfg.bUseShortSlotTime);

	if (pAd->CommonCfg.bWmmCapable)
	{
		// EDCA parameters used for AP's own transmission
		if (pAd->CommonCfg.APEdcaParm.bValid == FALSE)
		{
			pAd->CommonCfg.APEdcaParm.bValid = TRUE;
			pAd->CommonCfg.APEdcaParm.Aifsn[0] = 3;
			pAd->CommonCfg.APEdcaParm.Aifsn[1] = 7;
			pAd->CommonCfg.APEdcaParm.Aifsn[2] = 1;
			pAd->CommonCfg.APEdcaParm.Aifsn[3] = 1;

			pAd->CommonCfg.APEdcaParm.Cwmin[0] = 4;
			pAd->CommonCfg.APEdcaParm.Cwmin[1] = 4;
			pAd->CommonCfg.APEdcaParm.Cwmin[2] = 3;
			pAd->CommonCfg.APEdcaParm.Cwmin[3] = 2;

			pAd->CommonCfg.APEdcaParm.Cwmax[0] = 10;
			pAd->CommonCfg.APEdcaParm.Cwmax[1] = 6;
			pAd->CommonCfg.APEdcaParm.Cwmax[2] = 4;
			pAd->CommonCfg.APEdcaParm.Cwmax[3] = 3;

			pAd->CommonCfg.APEdcaParm.Txop[0]  = 0;
			pAd->CommonCfg.APEdcaParm.Txop[1]  = 0;
			pAd->CommonCfg.APEdcaParm.Txop[2]  = 96;
			pAd->CommonCfg.APEdcaParm.Txop[3]  = 48;
		}
		AsicSetEdcaParm(pAd, &pAd->CommonCfg.APEdcaParm);

		// EDCA parameters to be annouced in outgoing BEACON, used by WMM STA
		if (pAd->ApCfg.BssEdcaParm.bValid == FALSE)
		{
			pAd->ApCfg.BssEdcaParm.bValid = TRUE;
			pAd->ApCfg.BssEdcaParm.Aifsn[0] = 3;
			pAd->ApCfg.BssEdcaParm.Aifsn[1] = 7;
			pAd->ApCfg.BssEdcaParm.Aifsn[2] = 2;
			pAd->ApCfg.BssEdcaParm.Aifsn[3] = 2;

			pAd->ApCfg.BssEdcaParm.Cwmin[0] = 4;
			pAd->ApCfg.BssEdcaParm.Cwmin[1] = 4;
			pAd->ApCfg.BssEdcaParm.Cwmin[2] = 3;
			pAd->ApCfg.BssEdcaParm.Cwmin[3] = 2;

			pAd->ApCfg.BssEdcaParm.Cwmax[0] = 10;
			pAd->ApCfg.BssEdcaParm.Cwmax[1] = 10;
			pAd->ApCfg.BssEdcaParm.Cwmax[2] = 4;
			pAd->ApCfg.BssEdcaParm.Cwmax[3] = 3;

			pAd->ApCfg.BssEdcaParm.Txop[0]  = 0;
			pAd->ApCfg.BssEdcaParm.Txop[1]  = 0;
			pAd->ApCfg.BssEdcaParm.Txop[2]  = 96;
			pAd->ApCfg.BssEdcaParm.Txop[3]  = 48;
		}
	}
	else
		AsicSetEdcaParm(pAd, NULL);

	COPY_MAC_ADDR(pAd->CommonCfg.Bssid, pAd->CurrentAddress);
	AsicSetBssid(pAd, pAd->CurrentAddress);
	AsicSwitchChannel(pAd, pAd->CommonCfg.Channel);
	AsicLockChannel(pAd, pAd->CommonCfg.Channel);
	MlmeSetTxPreamble(pAd, (USHORT)pAd->CommonCfg.TxPreamble);
	MlmeUpdateTxRates(pAd, FALSE);

	// start sending BEACON out
	APMakeBssBeacon(pAd);
	APUpdateBeaconFrame(pAd);
	APUpdateCapabilityAndErpIe(pAd);
	if ((pAd->CommonCfg.PhyMode == PHY_11A) && (pAd->CommonCfg.bIEEE80211H == 1) && RadarChannelCheck(pAd, pAd->CommonCfg.Channel))
	{
		pAd->CommonCfg.RadarDetect.RDMode = RD_SILENCE_MODE;
		pAd->CommonCfg.RadarDetect.RDCount = 0;
		pAd->CommonCfg.RadarDetect.InServiceMonitorCount = 0;
		RadarDetectionStart(pAd);
	}
	else
	{
		AsicEnableBssSync(pAd);
	}

	// Set LED
	RTMPSetLED(pAd, LED_LINK_UP);

	// if AuthMode >= WPA,then pairwise key table must be used;
	// if legacy WEP inused, then only shared key table is used
	if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
	{
		DBGPRINT(RT_DEBUG_TRACE, ("AP AuthMode=%d, Pairwise Key Table in-used\n", pAd->CommonCfg.AuthMode));
		RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x0000000f);
	}
	else
	{
		DBGPRINT(RT_DEBUG_TRACE, ("AP AuthMode=%d, disable Pairwise Key Table\n", pAd->CommonCfg.AuthMode));
		RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x00000000);
	}

	DBGPRINT(RT_DEBUG_TRACE, (" APStartUp :  Group rekey method= %d , interval = 0x%x\n",pAd->ApCfg.WPAREKEY.ReKeyMethod,pAd->ApCfg.WPAREKEY.ReKeyInterval));
	// Group rekey related
	if ((pAd->ApCfg.WPAREKEY.ReKeyInterval != 0) && ((pAd->ApCfg.WPAREKEY.ReKeyMethod == TIME_REKEY) || (pAd->ApCfg.WPAREKEY.ReKeyMethod == PKT_REKEY)))
	{
		// Regularly check the timer
		if (pAd->ApCfg.REKEYTimerRunning == FALSE)
		{
#ifdef WIN_NDIS
			NdisMSetPeriodicTimer(&pAd->ApCfg.REKEYTimer, GROUP_KEY_UPDATE_EXEC_INTV);
#else
            /* ~~sample, modification for TIME_REKEY method */
            RTMPSetTimer(&pAd->ApCfg.REKEYTimer, GROUP_KEY_UPDATE_EXEC_INTV);
#endif
			pAd->ApCfg.REKEYTimerRunning = TRUE;
			pAd->ApCfg.REKEYCOUNTER = 0;
		}
	}


	// remove pairwise key table
	for (i=0; i<PAIRWISE_KEY_NUM; i++)
	{
		AsicRemovePairwiseKeyEntry(pAd,(UCHAR)  i);
	}

	// Init TKIP Group-Key-related variables
	APGenRandom(pAd,pAd->ApCfg.GMK);
	APGenRandom(pAd,pAd->ApCfg.GNonce);
	CountGTK(pAd->ApCfg.GMK, (UCHAR*)pAd->ApCfg.GNonce, pAd->CurrentAddress, GTK, TKIP_GTK_LENGTH);
	if ((pAd->CommonCfg.WepStatus == Ndis802_11Encryption2Enabled) ||
		(pAd->CommonCfg.WepStatus == Ndis802_11Encryption3Enabled) ||
		(pAd->CommonCfg.WepStatus == Ndis802_11Encryption4Enabled))
	{
		PCIPHER_KEY pKey = &pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId];
		pKey->KeyLen = LEN_TKIP_EK;
		NdisMoveMemory(pKey->Key, GTK, LEN_TKIP_EK);
		NdisMoveMemory(pKey->TxMic, &GTK[16], LEN_TKIP_TXMICK);
		NdisMoveMemory(pKey->RxMic, &GTK[24], LEN_TKIP_RXMICK);

		if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption3Enabled)
			pKey->CipherAlg = CIPHER_AES;
		else
			pKey->CipherAlg = CIPHER_TKIP;

		// install Group Key to MAC ASIC
		AsicAddSharedKeyEntry(
			pAd,
			BSS0,
			pAd->CommonCfg.DefaultKeyId,
			pKey->CipherAlg,
			pKey->Key,
			MAX_LEN_OF_SHARE_KEY,
			pKey->TxMic,
			pKey->RxMic);
	}

	ApLogEvent(pAd, pAd->CurrentAddress, EVENT_RESET_ACCESS_POINT);
	pAd->Mlme.PeriodicRound = 0;

	OPSTATUS_SET_FLAG(pAd, fOP_STATUS_MEDIA_STATE_CONNECTED);
	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_MEDIA_STATE_CHANGE);
	pAd->IndicateMediaState = NdisMediaStateConnected;

	RTMPSetTimer(&pAd->ApCfg.TBTTTimer, pAd->CommonCfg.BeaconPeriod);
	DBGPRINT(RT_DEBUG_TRACE, ("<=== APStartUp\n"));
}

/*
	==========================================================================
	Description:
		disassociate all STAs and stop AP service.
	Note:
	==========================================================================
 */
VOID APStop(
	IN PRTMP_ADAPTER pAd)
{
	BOOLEAN     Cancelled;

	DBGPRINT(RT_DEBUG_TRACE, ("!!! APStop !!!\n"));

	MacTableReset(pAd);

	if (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
	{
		//NICDisableInterrupt(pAd);
		AsicDisableSync(pAd);
		// Set LED
		RTMPSetLED(pAd, LED_LINK_DOWN);
	}

	if(pAd->ApCfg.REKEYTimerRunning==TRUE)
	{
#ifdef WIN_NDIS
		NdisMCancelTimer(&pAd->ApCfg.REKEYTimer, &Cancelled);
#else
		RTMPCancelTimer(&pAd->ApCfg.REKEYTimer, &Cancelled);
#endif
		pAd->ApCfg.REKEYTimerRunning=FALSE;
	}

	//
	// Cancel the Timer, to make sure the timer was not queued.
	//
#ifdef WIN_NDIS
	NdisMCancelTimer(&pAd->ApCfg.CounterMeasureTimer, &Cancelled);
	NdisMCancelTimer(&pAd->ApCfg.ApQuickResponeForRateUpTimer, &Cancelled);
#else
	RTMPCancelTimer(&pAd->ApCfg.CounterMeasureTimer, &Cancelled);
	RTMPCancelTimer(&pAd->ApCfg.ApQuickResponeForRateUpTimer, &Cancelled);
	RTMPCancelTimer(&pAd->ApCfg.TBTTTimer, &Cancelled);
#endif

	pAd->IndicateMediaState = NdisMediaStateDisconnected;
#ifdef WIN_NDIS
	NdisMIndicateStatus(pAd->AdapterHandle, NDIS_STATUS_MEDIA_DISCONNECT, (PVOID)NULL, 0);
	NdisMIndicateStatusComplete(pAd->AdapterHandle);
#endif
}

/*
	==========================================================================
	Description:
		This routine is used to clean up a specified power-saving queue. It's
		used whenever a wireless client is deleted.
	==========================================================================
 */
VOID APCleanupPsQueue(
	IN  PRTMP_ADAPTER   pAd,
	IN  PQUEUE_HEADER   pQueue)
{
	PQUEUE_ENTRY pEntry;
	PNDIS_PACKET pPacket;

	while (pQueue->Head)
	{
		pEntry = RemoveHeadQueue(pQueue);
		//pPacket = CONTAINING_RECORD(pEntry, NDIS_PACKET, MiniportReservedEx);
		pPacket = QUEUE_ENTRY_TO_PACKET(pEntry);
		RELEASE_NDIS_PACKET(pAd, pPacket, NDIS_STATUS_FAILURE);
	}
}

/*
	==========================================================================
	Description:
		This routine reset the entire MAC table. All packets pending in
		the power-saving queues are freed here.
	==========================================================================
 */
VOID MacTableReset(
	IN  PRTMP_ADAPTER  pAd)
{
	int         i;
	BOOLEAN     Cancelled;
	PCHAR       pOutBuffer = NULL;
	NDIS_STATUS NStatus;
	ULONG       FrameLen = 0;
	HEADER_802_11 DisassocHdr;
	USHORT      Reason;

#ifdef WIN_NDIS
	NdisAcquireSpinLock(&pAd->MacTabLock);
#else
	RTMP_SEM_LOCK(&pAd->MacTabLock);
#endif
	APCleanupPsQueue(pAd, &pAd->MacTab.McastPsQueue);
	for (i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
	{
		if (pAd->MacTab.Content[i].Valid)
		{
			// del EnqueueStartForPSKTimer timer.
			if (pAd->MacTab.Content[i].EnqueueStartForPSKTimerRunning == TRUE)
			{
				DBGPRINT(RT_DEBUG_TRACE, ("pAd->MacTab.Content[%d]  Cancel EnqueueStartForPSK Timer !\n",i));
#ifdef WIN_NDIS
				NdisMCancelTimer(&pAd->MacTab.Content[i].EnqueueStartForPSKTimer, &Cancelled);
#else
				RTMPCancelTimer(&pAd->MacTab.Content[i].EnqueueStartForPSKTimer, &Cancelled);
#endif
				pAd->MacTab.Content[i].EnqueueStartForPSKTimerRunning = FALSE;
			}

#ifdef WIN_NDIS
			NdisMCancelTimer(&pAd->MacTab.Content[i].RetryTimer, &Cancelled);
#else
			RTMPCancelTimer(&pAd->MacTab.Content[i].RetryTimer, &Cancelled);
#endif
			APCleanupPsQueue(pAd, &pAd->MacTab.Content[i].PsQueue);

			// Before reset MacTable, send disassociation packet to client.
			if ((pAd->MacTab.Content[i].Sst == SST_ASSOC)  && !RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
			{
				//  send out a DISASSOC request frame
				NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
				if (NStatus != NDIS_STATUS_SUCCESS)
					return;

				Reason = REASON_DISASSOC_INACTIVE;
				DBGPRINT(RT_DEBUG_ERROR, ("ASSOC - Send DISASSOC  Reason = %d frame  TO %x %x %x %x %x %x \n",Reason,pAd->MacTab.Content[i].Addr[0],
					pAd->MacTab.Content[i].Addr[1],pAd->MacTab.Content[i].Addr[2],pAd->MacTab.Content[i].Addr[3],pAd->MacTab.Content[i].Addr[4],pAd->MacTab.Content[i].Addr[5]));
				MgtMacHeaderInit(pAd, &DisassocHdr, SUBTYPE_DISASSOC, 0, pAd->MacTab.Content[i].Addr, pAd->CommonCfg.Bssid);
				MakeOutgoingFrame(pOutBuffer, &FrameLen, sizeof(HEADER_802_11), &DisassocHdr, 2, &Reason, END_OF_ARGS);
				MiniportMMRequest(pAd, pOutBuffer, FrameLen);

				RTMPusecDelay(2000);
			}
		}
	}
	NdisZeroMemory(&pAd->MacTab, sizeof(MAC_TABLE));
	InitializeQueueHeader(&pAd->MacTab.McastPsQueue);
#ifdef WIN_NDIS
	NdisReleaseSpinLock(&pAd->MacTabLock);
#else
	RTMP_SEM_UNLOCK(&pAd->MacTabLock);
#endif
}

/*
	==========================================================================
	Description:
		Add and new entry into MAC table
	==========================================================================
 */
MAC_TABLE_ENTRY *MacTableInsertEntry(
	IN  PRTMP_ADAPTER   pAd,
	IN  PUCHAR pAddr)
{
	UCHAR HashIdx;
	int i;
	MAC_TABLE_ENTRY *pEntry = NULL, *pCurrEntry;

	// if FULL, return
	if (pAd->MacTab.Size >= MAX_LEN_OF_MAC_TABLE)
		return NULL;

	// allocate one MAC entry
#ifdef WIN_NDIS
	NdisAcquireSpinLock(&pAd->MacTabLock);
#else
	RTMP_SEM_LOCK(&pAd->MacTabLock);
#endif

	for (i = 1; i< MAX_LEN_OF_MAC_TABLE; i++)   // skip entry#0 so that "entry index == AID" for fast lookup
	{
		// pick up the first available vacancy
		if (pAd->MacTab.Content[i].Valid == FALSE)
		{
			pEntry = &pAd->MacTab.Content[i];
			NdisZeroMemory(pEntry, sizeof(MAC_TABLE_ENTRY));
			pEntry->Valid = TRUE;
#ifdef WIN_NDIS
			NdisMInitializeTimer(&pEntry->RetryTimer, pAd->AdapterHandle, WPARetryExec, pEntry);
			NdisMInitializeTimer(&pEntry->EnqueueStartForPSKTimer, pAd->AdapterHandle, WPARetryExec, pEntry);
#else
			RTMPInitTimer(pAd, &pEntry->RetryTimer,GET_TIMER_FUNCTION(WPARetryExec), pEntry, TRUE);
			RTMPInitTimer(pAd, &pEntry->EnqueueStartForPSKTimer, GET_TIMER_FUNCTION(EnqueueStartForPSKExec), pEntry, FALSE);
#endif

			pEntry->pAd = pAd;
			pEntry->CMTimerRunning = FALSE;
			pEntry->EnqueueStartForPSKTimerRunning= FALSE;
			pEntry->RSNIE_Len = 0;
			NdisZeroMemory(pEntry->R_Counter, sizeof(pEntry->R_Counter));
			pEntry->ReTryCounter = PEER_MSG1_RETRY_TIMER_CTR;;
			pEntry->AuthMode = pAd->CommonCfg.AuthMode;
			pEntry->WepStatus = pAd->CommonCfg.WepStatus;

			if (pEntry->AuthMode < Ndis802_11AuthModeWPA)
				pEntry->WpaState = AS_NOTUSE;
			else
				pEntry->WpaState = AS_INITIALIZE;
			pEntry->GTKState = REKEY_NEGOTIATING;
			pEntry->PrivacyFilter = Ndis802_11PrivFilterAcceptAll;
			pEntry->PairwiseKey.KeyLen = 0;
			pEntry->PairwiseKey.CipherAlg = CIPHER_NONE;
			pEntry->PortSecured = WPA_802_1X_PORT_NOT_SECURED;
			RTUSBEnqueueInternalCmd(pAd, RT_OID_ASIC_REMOVE_PKEY, NULL, 0);
			//AsicRemovePairwiseKeyEntry(pAd, (UCHAR)i); //TO BE CHECK;
			COPY_MAC_ADDR(pEntry->Addr, pAddr);
			pEntry->Sst = SST_NOT_AUTH;
			pEntry->AuthState = AS_NOT_AUTH;
			pEntry->Aid = (USHORT)i;  //0;
			pEntry->CapabilityInfo = 0;
			pEntry->PsMode = PWR_ACTIVE;
			pEntry->MaxSupportedRate = RATE_11;
			pEntry->CurrTxRate = RATE_11;
			pEntry->PsQIdleCount = 0;
			pEntry->NoDataIdleCount = 0;
			InitializeQueueHeader(&pEntry->PsQueue);
			pAd->MacTab.Size ++;

			DBGPRINT(RT_DEBUG_TRACE, ("MacTableInsertEntry - allocate entry #%d, Total= %d\n",i, pAd->MacTab.Size));
			break;
		}
	}

	// add this MAC entry into HASH table
	if (pEntry)
	{
		HashIdx = MAC_ADDR_HASH_INDEX(pAddr);
		if (pAd->MacTab.Hash[HashIdx] == NULL)
		{
			pAd->MacTab.Hash[HashIdx] = pEntry;
		}
		else
		{
			pCurrEntry = pAd->MacTab.Hash[HashIdx];
			while (pCurrEntry->pNext != NULL)
				pCurrEntry = pCurrEntry->pNext;
			pCurrEntry->pNext = pEntry;
		}
	}
#ifdef WIN_NDIS
	NdisReleaseSpinLock(&pAd->MacTabLock);
#else
	RTMP_SEM_UNLOCK(&pAd->MacTabLock);
#endif
	return pEntry;
}

/*
	==========================================================================
	Description:
		Delete a specified client from MAC table
	==========================================================================
 */
BOOLEAN MacTableDeleteEntry(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR pAddr)
{
	USHORT HashIdx;
	MAC_TABLE_ENTRY *pEntry, *pPrevEntry;
	BOOLEAN Cancelled;
	UCHAR idx;

#ifdef WIN_NDIS
	NdisAcquireSpinLock(&pAd->MacTabLock);
#else
	RTMP_SEM_LOCK(&pAd->MacTabLock);
#endif
	HashIdx = MAC_ADDR_HASH_INDEX(pAddr);
	pEntry = pAd->MacTab.Hash[HashIdx];

	if (pEntry)
	{
		if (MAC_ADDR_EQUAL(pEntry->Addr, pAddr))
		{
			//no piggyback support
			/*if (CLIENT_STATUS_TEST_FLAG(pEntry, fCLIENT_STATUS_PIGGYBACK_CAPABLE))
			{
				pAd->ApCfg.bOneClientUsePiggyBack = FALSE;
				//RTMPSetPiggyBack(pAd, pEntry, FALSE);
			}*/
			idx = (UCHAR)pEntry->Aid;
			RTUSBEnqueueInternalCmd(pAd, RT_OID_ASIC_REMOVE_PKEY, NULL, 0);
			pAd->MacTab.Hash[HashIdx] = pEntry->pNext;
			APCleanupPsQueue(pAd, &pEntry->PsQueue); // return all NDIS packet in PSQ

			if(pEntry->EnqueueStartForPSKTimerRunning == TRUE)
			{
				DBGPRINT(RT_DEBUG_TRACE, ("MacTableDeleteEntry : Cancel EnqueueStartForPSK Timer !\n"));
				RTMPCancelTimer(&pEntry->EnqueueStartForPSKTimer, &Cancelled);
				pEntry->EnqueueStartForPSKTimerRunning = FALSE;
			}

			NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
			NdisZeroMemory(pEntry, sizeof(MAC_TABLE_ENTRY));
			pAd->MacTab.Size --;
			DBGPRINT(RT_DEBUG_TRACE, ("MacTableDeleteEntry1 - Total= %d\n", pAd->MacTab.Size));
		}
		else
		{
			while (pEntry->pNext)
			{
				pPrevEntry = pEntry;
				pEntry = pEntry->pNext;
				if (MAC_ADDR_EQUAL(pEntry->Addr, pAddr))
				{
					/*if (CLIENT_STATUS_TEST_FLAG(pEntry, fCLIENT_STATUS_PIGGYBACK_CAPABLE))
					{
						pAd->ApCfg.bOneClientUsePiggyBack = FALSE;
						//RTMPSetPiggyBack(pAd, pEntry, FALSE);
					}*/
					idx = (UCHAR)pEntry->Aid;
					RTUSBEnqueueInternalCmd(pAd, RT_OID_ASIC_REMOVE_PKEY, NULL, 0);
					pPrevEntry->pNext = pEntry->pNext;
					APCleanupPsQueue(pAd, &pEntry->PsQueue); // return all NDIS packet in PSQ

					if(pEntry->EnqueueStartForPSKTimerRunning == TRUE)
					{
						DBGPRINT(RT_DEBUG_TRACE, ("MacTableDeleteEntry : Cancel EnqueueStartForPSK Timer !\n"));
						RTMPCancelTimer(&pEntry->EnqueueStartForPSKTimer, &Cancelled);
						pEntry->EnqueueStartForPSKTimerRunning = FALSE;
					}

					NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
					NdisZeroMemory(pEntry, sizeof(MAC_TABLE_ENTRY));
					pAd->MacTab.Size --;
					DBGPRINT(RT_DEBUG_TRACE, ("MacTableDeleteEntry2 - Total= %d\n", pAd->MacTab.Size));
					break;
				}
			}
		}
	}
#ifdef WIN_NDIS
	NdisReleaseSpinLock(&pAd->MacTabLock);
#else
	RTMP_SEM_UNLOCK(&pAd->MacTabLock);
#endif
	//don't call this function ,will cause crash in dispacth level
	//APUpdateCapabilityAndErpIe(pAd);

	return TRUE;
}

/*
	==========================================================================
	Description:
		Look up the MAC address in the MAC table. Return NULL if not found.
	Return:
		pEntry - pointer to the MAC entry; NULL is not found
	==========================================================================
*/
MAC_TABLE_ENTRY *MacTableLookup(
	IN PRTMP_ADAPTER pAd,
	PUCHAR pAddr)
{
	ULONG HashIdx;
	MAC_TABLE_ENTRY *pEntry;

	HashIdx = MAC_ADDR_HASH_INDEX(pAddr);
	pEntry = pAd->MacTab.Hash[HashIdx];

	while (pEntry && pEntry->Valid)
	{
		if (MAC_ADDR_EQUAL(pEntry->Addr, pAddr))
		{
			break;
		}
		else
			pEntry = pEntry->pNext;
	}
	return pEntry;
}

/*
	==========================================================================
	Description:
		This routine is called by APMlmePeriodicExec() every second to check if
		1. any associated client in PSM. If yes, then TX MCAST/BCAST should be
		   out in DTIM only
		2. any client being idle for too long and should be aged-out from MAC table
		3. garbage collect PSQ
	==========================================================================
*/
VOID MacTableMaintenance(
	IN PRTMP_ADAPTER pAd)
{
	int i;

	pAd->MacTab.fAnyStationInPsm = FALSE;
	for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
	{
		MAC_TABLE_ENTRY *pEntry = &pAd->MacTab.Content[i];

		if (pEntry->Valid == FALSE)
			continue;

		pEntry->NoDataIdleCount ++;

		// 0. STA failed to complete association should be removed to save MAC table space.
		if ((pEntry->Sst != SST_ASSOC) && (pEntry->NoDataIdleCount >= MAC_TABLE_ASSOC_TIMEOUT))
		{
			DBGPRINT(RT_DEBUG_TRACE, ("%02x:%02x:%02x:%02x:%02x:%02x fail to complete ASSOC in %d sec\n",
					pEntry->Addr[0],pEntry->Addr[1],pEntry->Addr[2],pEntry->Addr[3],
					pEntry->Addr[4],pEntry->Addr[5],MAC_TABLE_ASSOC_TIMEOUT));
			MacTableDeleteEntry(pAd, pEntry->Addr);
			continue;
		}

		// 1. check if there's any associated STA in power-save mode. this affects outgoing
		//    MCAST/BCAST frames should be stored in PSQ till DtimCount=0
		if (pEntry->PsMode == PWR_SAVE)
			pAd->MacTab.fAnyStationInPsm = TRUE;

		// 2. delete those MAC entry that has been idle for a long time
		if ((pEntry->NoDataIdleCount >= pAd->ApCfg.AgeoutTime) && (pAd->ApCfg.AgeoutTime != 0))
		{
			DBGPRINT(RT_DEBUG_TRACE, ("ageout %02x:%02x:%02x:%02x:%02x:%02x after %d-sec silence\n",
					pEntry->Addr[0],pEntry->Addr[1],pEntry->Addr[2],pEntry->Addr[3],
					pEntry->Addr[4],pEntry->Addr[5],pAd->ApCfg.AgeoutTime));
			ApLogEvent(pAd, pEntry->Addr, EVENT_AGED_OUT);
			MacTableDeleteEntry(pAd, pEntry->Addr);
			continue;
		}

		// 3. garbage collect the PsQueue if the STA has being idle for a while
		if (pEntry->PsQueue.Head)
		{
			pEntry->PsQIdleCount ++;
			if (pEntry->PsQIdleCount > 2)
			{
#ifdef WIN_NDIS
				NdisAcquireSpinLock(&pAd->MacTabLock);
#else
				RTMP_SEM_LOCK(&pAd->MacTabLock);
#endif
				APCleanupPsQueue(pAd, &pEntry->PsQueue);
#ifdef WIN_NDIS
				NdisReleaseSpinLock(&pAd->MacTabLock);
#else
				RTMP_SEM_UNLOCK(&pAd->MacTabLock);
#endif

				pEntry->PsQIdleCount = 0;
			}
		}
		else
			pEntry->PsQIdleCount = 0;
	}

	// 4. garbage collect pAd->MacTab.McastPsQueue if backlogged MCAST/BCAST frames
	//    stale in queue. Since MCAST/BCAST frames always been sent out whenever
	//    DtimCount==0, the only case to let them stale is surprise removal of the NIC,
	//    so that ASIC-based Tbcn interrupt stops and DtimCount dead.
	if (pAd->MacTab.McastPsQueue.Head)
	{
		pAd->MacTab.PsQIdleCount ++;
		if (pAd->MacTab.PsQIdleCount > 1)
		{
#ifdef WIN_NDIS
			NdisAcquireSpinLock(&pAd->MacTabLock);
#else
			RTMP_SEM_LOCK(&pAd->MacTabLock);
#endif

			APCleanupPsQueue(pAd, &pAd->MacTab.McastPsQueue);
#ifdef WIN_NDIS
			NdisReleaseSpinLock(&pAd->MacTabLock);
#else
			RTMP_SEM_UNLOCK(&pAd->MacTabLock);
#endif

			pAd->MacTab.PsQIdleCount = 0;
		}
	}
	else
		pAd->MacTab.PsQIdleCount = 0;
}

WDS_ENTRY *WdsTableInsertEntry(
	IN  PRTMP_ADAPTER   pAd,
	IN  PUCHAR          pAddr,
	IN  ULONG           index)
{
	UCHAR HashIdx;
	int i;
	WDS_ENTRY *pEntry = NULL, *pCurrEntry;

	if (pAd->WdsTab.MacTab[index].Size >= MAX_LEN_OF_MAC_TABLE)
		return NULL;

	// allocate one MAC entry
#ifdef WIN_NDIS
	NdisAcquireSpinLock(&pAd->WdsTabLock);
#else
	RTMP_SEM_LOCK(&pAd->WdsTabLock);
#endif

	for (i = 0; i< MAX_LEN_OF_MAC_TABLE; i++)
	{
		// pick up the first available vacancy
		if (pAd->WdsTab.MacTab[index].Content[i].Valid == FALSE)
		{
			pEntry = &pAd->WdsTab.MacTab[index].Content[i];
			NdisZeroMemory(pEntry, sizeof(WDS_ENTRY));
			pEntry->Valid = TRUE;
			COPY_MAC_ADDR(pEntry->Addr, pAddr);
			pEntry->NoDataIdleCount = 0;
			pAd->WdsTab.MacTab[index].Size ++;
			DBGPRINT(RT_DEBUG_TRACE, ("WdsTableInsertEntry - allocate entry #%d, Total= %d\n",i, pAd->WdsTab.MacTab[index].Size));
			break;
		}
	}

	// add this MAC entry into HASH table
	if (pEntry)
	{
		HashIdx = MAC_ADDR_HASH_INDEX(pAddr);
		if (pAd->WdsTab.MacTab[index].Hash[HashIdx] == NULL)
		{
			pAd->WdsTab.MacTab[index].Hash[HashIdx] = pEntry;
		}
		else
		{
			pCurrEntry = pAd->WdsTab.MacTab[index].Hash[HashIdx];
			while (pCurrEntry->pNext != NULL)
				pCurrEntry = pCurrEntry->pNext;
			pCurrEntry->pNext = pEntry;
		}
	}
#ifdef WIN_NDIS
	NdisReleaseSpinLock(&pAd->WdsTabLock);
#else
	RTMP_SEM_UNLOCK(&pAd->WdsTabLock);
#endif
	return pEntry;
}

BOOLEAN WdsTableDeleteEntry(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR        pAddr,
	IN ULONG         index)
{
	USHORT HashIdx;
	WDS_ENTRY *pEntry, *pPrevEntry;
#ifdef WIN_NDIS
	NdisAcquireSpinLock(&pAd->WdsTabLock);
#else
	RTMP_SEM_LOCK(&pAd->WdsTabLock);
#endif
	HashIdx = MAC_ADDR_HASH_INDEX(pAddr);
	pEntry = pAd->WdsTab.MacTab[index].Hash[HashIdx];
	if (pEntry)
	{
		if (MAC_ADDR_EQUAL(pEntry->Addr, pAddr))
		{
			pAd->WdsTab.MacTab[index].Hash[HashIdx] = pEntry->pNext;
			NdisZeroMemory(pEntry, sizeof(WDS_ENTRY));
			pAd->WdsTab.MacTab[index].Size --;
			DBGPRINT(RT_DEBUG_TRACE, ("WdsTableDeleteEntry - Total= %d\n", pAd->WdsTab.MacTab[index].Size));
		}
		else
		{
			while (pEntry->pNext)
			{
				pPrevEntry = pEntry;
				pEntry = pEntry->pNext;
				if (MAC_ADDR_EQUAL(pEntry->Addr, pAddr))
				{
					pPrevEntry->pNext = pEntry->pNext;
					NdisZeroMemory(pEntry, sizeof(WDS_ENTRY));
					pAd->WdsTab.MacTab[index].Size --;
					DBGPRINT(RT_DEBUG_TRACE, ("WdsTableDeleteEntry - Total= %d\n", pAd->WdsTab.MacTab[index].Size));
					break;
				}
			}
		}
	}
#ifdef WIN_NDIS
	NdisReleaseSpinLock(&pAd->WdsTabLock);
#else
	RTMP_SEM_UNLOCK(&pAd->WdsTabLock);
#endif

	return TRUE;
}

WDS_ENTRY *WdsTableLookup(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR        pAddr,
	IN ULONG         index)
{
	ULONG HashIdx;
	WDS_ENTRY *pEntry;

	HashIdx = MAC_ADDR_HASH_INDEX(pAddr);
	pEntry = pAd->WdsTab.MacTab[index].Hash[HashIdx];

	while (pEntry && pEntry->Valid)
	{
		if (MAC_ADDR_EQUAL(pEntry->Addr, pAddr))
		{
			pEntry->NoDataIdleCount = 0;
			break;
		}
		else
			pEntry = pEntry->pNext;
	}
	return pEntry;
}

/*
	==========================================================================
	Description:
		This routine is called by APMlmePeriodicExec() every second to check if
		1. any associated client in PSM. If yes, then TX MCAST/BCAST should be
		   out in DTIM only
		2. any client being idle for too long and should be aged-out from MAC table
		3. garbage collect PSQ
	==========================================================================
*/
VOID WdsTableMaintenance(
	IN PRTMP_ADAPTER pAd)
{
	ULONG   i, idx;

	for (idx = 0; idx < pAd->WdsTab.Size; idx++)
	{
		for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
		{
			WDS_ENTRY *pEntry = &pAd->WdsTab.MacTab[idx].Content[i];

			if (pEntry->Valid == FALSE)
				continue;

			pEntry->NoDataIdleCount += 5;

			// 2. delete those MAC entry that has been idle for a long time
			if ((pEntry->NoDataIdleCount >= pAd->ApCfg.AgeoutTime) && (pAd->ApCfg.AgeoutTime != 0))
			{
				DBGPRINT(RT_DEBUG_TRACE, ("ageout %02x:%02x:%02x:%02x:%02x:%02x from WDS #%d after %d-sec silence\n",
						pEntry->Addr[0],pEntry->Addr[1],pEntry->Addr[2],pEntry->Addr[3],
						pEntry->Addr[4],pEntry->Addr[5], idx, pAd->ApCfg.AgeoutTime));
				WdsTableDeleteEntry(pAd, pEntry->Addr, idx);
			}
		}
	}
}

/*
	==========================================================================
	Description:
		Look up a STA MAC table. Return its Sst to decide if an incoming
		frame from this STA or an outgoing frame to this STA is permitted.
	Return:
	==========================================================================
*/

MAC_TABLE_ENTRY *APSsPsInquiry(
	IN  PRTMP_ADAPTER   pAd,
	IN  PUCHAR pAddr,
	OUT SST   *Sst,
	OUT USHORT *Aid,
	OUT UCHAR *PsMode,
	OUT UCHAR *Rate)
{
	MAC_TABLE_ENTRY *pEntry = NULL;

	if (pAddr[0] & 0x01) // mcast & broadcast address
	{
		*Sst        = SST_ASSOC;
		*Aid        = 0;
		*PsMode     = PWR_ACTIVE;
		*Rate       = pAd->CommonCfg.MlmeRate;
	}
	else // unicast address
	{
		pEntry = MacTableLookup(pAd, pAddr);
		if (pEntry)
		{
			*Sst        = pEntry->Sst;
			*Aid        = pEntry->Aid;
			*PsMode     = pEntry->PsMode;
			if ((pEntry->AuthMode >= Ndis802_11AuthModeWPA) && (pEntry->GTKState != REKEY_ESTABLISHED))
				*Rate   = pAd->CommonCfg.MlmeRate;
			else
			*Rate       = pEntry->CurrTxRate;
		}
		else
		{
			*Sst        = SST_NOT_AUTH;
			*Aid        = 0;
			*PsMode     = PWR_ACTIVE;
			*Rate       = pAd->CommonCfg.MlmeRate;
		}
	}
	return pEntry;
}

/*
	==========================================================================
	Description:
		Update the station current power save mode. Calling this routine also
		prove the specified client is still alive. Otherwise AP will age-out
		this client once IdleCount exceeds a threshold.
	==========================================================================
 */
BOOLEAN APPsIndicate(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR pAddr,
	IN UCHAR Rssi,
	IN UCHAR Psm)
{
	MAC_TABLE_ENTRY *pEntry;

	pEntry = MacTableLookup(pAd, pAddr);
	if (pEntry)
	{
		if ((pEntry->PsMode == PWR_SAVE) && (Psm == PWR_ACTIVE))
		{
			DBGPRINT(RT_DEBUG_TRACE, ("APPsIndicate - %02x:%02x:%02x:%02x:%02x:%02x wakes up, act like rx PS-POLL\n", pAddr[0],pAddr[1],pAddr[2],pAddr[3],pAddr[4],pAddr[5]));
			// sleep station awakes, move all pending frames from PSQ to TXQ if any
			APHandleRxPsPoll(pAd, pAddr, pEntry->Aid, TRUE);
		}
		else if ((pEntry->PsMode != PWR_SAVE) && (Psm == PWR_SAVE))
		{
			DBGPRINT(RT_DEBUG_TRACE, ("APPsIndicate - %02x:%02x:%02x:%02x:%02x:%02x sleeps\n", pAddr[0],pAddr[1],pAddr[2],pAddr[3],pAddr[4],pAddr[5]));
		}

		pEntry->NoDataIdleCount = 0;
		pEntry->PsMode = Psm;
		pEntry->LastRssi = Rssi;
	}
	else
	{
		// not in table, try to learn it ???? why bother?
	}
	return TRUE;
}

/*
	==========================================================================
	Description:
		This routine is called to log a specific event into the event table.
		The table is a QUERY-n-CLEAR array that stop at full.
	==========================================================================
 */
VOID ApLogEvent(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR   pAddr,
	IN USHORT   Event)
{
	if (pAd->EventTab.Num < MAX_NUM_OF_EVENT)
	{
		RT_802_11_EVENT_LOG *pLog = &pAd->EventTab.Log[pAd->EventTab.Num];
		NdisGetCurrentSystemTime(&pLog->SystemTime);
		COPY_MAC_ADDR(pLog->Addr, pAddr);
		pLog->Event = Event;
		DBGPRINT_RAW(RT_DEBUG_TRACE,("LOG#%d %02x:%02x:%02x:%02x:%02x:%02x %s\n",
			pAd->EventTab.Num, pAddr[0], pAddr[1], pAddr[2],
			pAddr[3], pAddr[4], pAddr[5], pEventText[Event]));
		pAd->EventTab.Num += 1;
	}
}

/*
	==========================================================================
	Description:
		Update ERP IE and CapabilityInfo based on STA association status.
		The result will be auto updated into the next outgoing BEACON in next
		TBTT interrupt service routine
	==========================================================================
 */
VOID APUpdateCapabilityAndErpIe(
	IN PRTMP_ADAPTER pAd)
{
	UCHAR  i, ErpIeContent = 0;
	BOOLEAN ShortSlotCapable = pAd->CommonCfg.bUseShortSlotTime;
	static UCHAR lastErpIeContent;
	lastErpIeContent = pAd->ApCfg.ErpIeContent;
#if 0
	if (pAd->CommonCfg.PhyMode != PHY_11BG_MIXED)
	{
		if (pAd->CommonCfg.PhyMode == PHY_11G)
		{
			pAd->ApCfg.ErpIeContent = (ErpIeContent | 0x04);
			AsicSetSlotTime(pAd, ShortSlotCapable);
		}
		return;
	}
#else
	if (pAd->CommonCfg.PhyMode == PHY_11B)
		return;
#endif

	for (i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
	{
		PMAC_TABLE_ENTRY pEntry = &pAd->MacTab.Content[i];
		if ((pEntry->Valid != TRUE) || (pEntry->Sst != SST_ASSOC))
			continue;

		// at least one 11b client associated, turn on ERP.NonERPPresent bit
		// almost all 11b client won't support "Short Slot" time, turn off for maximum compatibility
		if (pEntry->MaxSupportedRate < RATE_FIRST_OFDM_RATE)
		{
			ShortSlotCapable = FALSE;
			ErpIeContent |= 0x01;
		}

		// at least one client can't support short slot
		if ((pEntry->CapabilityInfo & 0x0400) == 0)
			ShortSlotCapable = FALSE;
	}

	// decide ErpIR.UseProtection bit, depending on pAd->CommonCfg.UseBGProtection
	//    AUTO (0): UseProtection = 1 if any 11b STA associated
	//    ON (1): always USE protection
	//    OFF (2): always NOT USE protection
	if (pAd->CommonCfg.UseBGProtection == 0)
	{
		ErpIeContent = (ErpIeContent)? 0x03 : 0x00;

		if (RTMPConvertToMiniSecond(pAd->ApCfg.LastOLBCDetectTime, pAd->Mlme.Now64) < 5000)  // legacy BSS leave for 5 sec
		{
			DBGPRINT(RT_DEBUG_INFO, ("APUpdateCapabilityAndErpIe - Legacy 802.11b BSS overlaped\n"));
			ErpIeContent |= 0x02;                                     // set Use_Protection bit
		}
	}
	else if (pAd->CommonCfg.UseBGProtection == 1)
		ErpIeContent |= 0x02;
	else
		;

	pAd->ApCfg.ErpIeContent = (ErpIeContent | 0x04);

	//
	// deicide CapabilityInfo.ShortSlotTime bit
	//
	if (ShortSlotCapable)
		pAd->ApCfg.CapabilityInfo |= 0x0400;
	else
		pAd->ApCfg.CapabilityInfo &= 0xfbff;

	DBGPRINT(RT_DEBUG_INFO, ("APUpdateCapabilityAndErpIe - Capability= 0x%04x, ERP is 0x%02x\n",
		pAd->ApCfg.CapabilityInfo, ErpIeContent));

	AsicSetSlotTime(pAd, ShortSlotCapable);
	if ( pAd->ApCfg.ErpIeContent != lastErpIeContent)
      		pAd->ApCfg.bErpIEChange = TRUE;
}

/*
	==========================================================================
	Description:
		Check if the specified STA pass the Access Control List checking.
		If fails to pass the checking, then no authentication nor association
		is allowed
	Return:
		MLME_SUCCESS - this STA passes ACL checking

	==========================================================================
*/
BOOLEAN ApCheckAccessControlList(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR        pAddr)
{
	BOOLEAN Result = TRUE;

	if (pAd->AccessControlList.Policy == 0)       // ACL is disabled
		Result = TRUE;
	else
	{
		ULONG i;
		if (pAd->AccessControlList.Policy == 1)   // ACL is a positive list
			Result = FALSE;
		else                                              // ACL is a negative list
			Result = TRUE;
		for (i=0; i<pAd->AccessControlList.Num; i++)
		{
			if (MAC_ADDR_EQUAL(pAddr, pAd->AccessControlList.Entry[i].Addr))
			{
				Result = !Result;
				break;
			}
		}
	}

	if (Result == FALSE)
	{
		DBGPRINT(RT_DEBUG_TRACE, ("%02x:%02x:%02x:%02x:%02x:%02x failed ACL checking\n", pAddr[0],pAddr[1],pAddr[2],pAddr[3],pAddr[4],pAddr[5]));
	}

	return Result;
}

/*
	==========================================================================
	Description:
		This routine update the current MAC table based on the current ACL.
		If ACL change causing an associated STA become un-authorized. This STA
		wil be kicked out immediately.
	==========================================================================
*/
VOID ApUpdateAccessControlList(
	IN PRTMP_ADAPTER pAd)
{
	USHORT   AclIdx, MacIdx;
	BOOLEAN  Matched;

	// ACL is disabled. do nothing about the MAC table
	if (pAd->AccessControlList.Policy == 0)
		return;

	for (MacIdx=0; MacIdx < MAX_LEN_OF_MAC_TABLE; MacIdx++)
	{
		if (! pAd->MacTab.Content[MacIdx].Valid)
			continue;

		Matched = FALSE;
		for (AclIdx = 0; AclIdx < pAd->AccessControlList.Num; AclIdx++)
		{
			if (MAC_ADDR_EQUAL(pAd->MacTab.Content[MacIdx].Addr, pAd->AccessControlList.Entry[AclIdx].Addr))
			{
				Matched = TRUE;
				break;
			}
		}

		if ((Matched == FALSE) && (pAd->AccessControlList.Policy == 1))
		{
			DBGPRINT(RT_DEBUG_TRACE, ("STA not on positive ACL. remove it...\n"));
			MacTableDeleteEntry(pAd, pAd->MacTab.Content[MacIdx].Addr);
		}
		else if ((Matched == TRUE) && (pAd->AccessControlList.Policy == 2))
		{
			DBGPRINT(RT_DEBUG_TRACE, ("STA on negative ACL. remove it...\n"));
			MacTableDeleteEntry(pAd, pAd->MacTab.Content[MacIdx].Addr);
		}
	}
}

/*
	==========================================================================
	Description:
		Send out a NULL frame to a specified STA at a higher TX rate. The
		purpose is to ensure the designated client is okay to received at this
		rate.
	==========================================================================
 */
VOID ApEnqueueNullFrame(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR        pAddr,
	IN UCHAR         TxRate,
	IN UCHAR         PID)
{
	NDIS_STATUS    NState;
	PHEADER_802_11 pNullFr;

	// since TxRate may change, we have to change Duration each time
	NState = MlmeAllocateMemory(pAd, (PVOID)&pNullFr);
	if (NState == NDIS_STATUS_SUCCESS)
	{
		if ((PID & 0x3f) < WDS_PAIRWISE_KEY_OFFSET) // send to client
		{
			MgtMacHeaderInit(pAd, pNullFr, SUBTYPE_NULL_FUNC, 0, pAddr, pAd->CommonCfg.Bssid);
			pNullFr->FC.Type = BTYPE_DATA;
			pNullFr->FC.FrDs = 1;
			pNullFr->Duration = RTMPCalcDuration(pAd, TxRate, 14);
			DBGPRINT(RT_DEBUG_TRACE, ("send NULL Frame @%d Mbps to AID#%d...\n", RateIdToMbps[TxRate], PID & 0x3f));
			ApSendFrame(pAd, pNullFr, sizeof(HEADER_802_11), TxRate, PID);
		}
#ifdef  WDS
		else                                        // send to WDS link
		{
			UCHAR ToWhichWds = (PID & 0x3f) - WDS_PAIRWISE_KEY_OFFSET;
			NdisZeroMemory(pNullFr, LENGTH_802_11_WITH_ADDR4);
			pNullFr->FC.Type = BTYPE_DATA;
			pNullFr->FC.SubType = SUBTYPE_NULL_FUNC;
			pNullFr->FC.FrDs = 1;
			pNullFr->FC.ToDs = 1;
			COPY_MAC_ADDR(pNullFr->Addr1, pAd->WdsTab.MacTab[ToWhichWds].WdsAddr);
			COPY_MAC_ADDR(pNullFr->Addr2, pAd->CurrentAddress);
			COPY_MAC_ADDR(pNullFr->Addr3, pAd->WdsTab.MacTab[ToWhichWds].WdsAddr);
			COPY_MAC_ADDR(pNullFr->Addr3 + MAC_ADDR_LEN, pAd->CurrentAddress);
			pNullFr->Duration = RTMPCalcDuration(pAd, TxRate, 14);
			DBGPRINT(RT_DEBUG_TRACE, ("send NULL Frame @%d Mbps to WDS#%d...\n", RateIdToMbps[TxRate], ToWhichWds));
			ApSendFrame(pAd, pNullFr, LENGTH_802_11_WITH_ADDR4, TxRate, PID);
		}
#endif
	MlmeFreeMemory(pAd, pNullFr);
	}
}

VOID    ApSendFrame(
	IN  PRTMP_ADAPTER   pAd,
	IN  PVOID           pBuffer,
	IN  ULONG           Length,
	IN  UCHAR           TxRate,
	IN  UCHAR           PID)
{
	PUCHAR          pDest;
	PTXD_STRUC      pTxD;
	ULONG			TransferBufferLength;
	PTX_CONTEXT		pMLMEContext;
	//PRTMP_TX_RING   pTxRing = &pAd->TxRing[QID_AC_BE];

	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS))
	{
		return;
	}



	pMLMEContext = &pAd->MLMEContext[pAd->NextMLMEIndex];
	pMLMEContext->InUse = TRUE;

	// Increase & maintain Tx Ring Index
	pAd->NextMLMEIndex++;
	if (pAd->NextMLMEIndex >= PRIO_RING_SIZE)
	{
		pAd->NextMLMEIndex = 0;
	}

	// Make sure Tx ring resource won't be used by other threads
	//NdisAcquireSpinLock(&pAd->TxRingLock);

	// Get the Tx Ring descriptor & Dma Buffer address
	pDest = pMLMEContext->TransferBuffer->u.WirelessPacket;

	pTxD = (PTXD_STRUC)(&pMLMEContext->TransferBuffer->TxDesc);

	NdisMoveMemory(pDest, pBuffer, Length);
//	pTxRing->Cell[pTxRing->CurTxIndex].FrameType = BTYPE_DATA;

	if (((PID & 0x3f) < WDS_PAIRWISE_KEY_OFFSET) && ((PID & 0x3f) > 0))
	{
		MAC_TABLE_ENTRY *pEntry;

		pEntry = &pAd->MacTab.Content[(PID & 0x3f)];
	}



	 RTUSBWriteTxDescriptor(pAd,
						  pTxD,
						  CIPHER_NONE, 0, 0,
						  TRUE,
						  FALSE,
						  FALSE,
						  SHORT_RETRY,
						  IFS_BACKOFF,
						  TxRate,
						  Length,
						  QID_AC_BE,
						  PID,
						  FALSE);

	//INC_RING_INDEX(pTxRing->CurTxIndex, TX_RING_SIZE);
	pAd->RalinkCounters.KickTxCount++;
	TransferBufferLength = sizeof(TXD_STRUC) + Length;
	if ((TransferBufferLength % 2) == 1)
		TransferBufferLength++;

	if ((TransferBufferLength % pAd->BulkOutMaxPacketSize) == 0)
		TransferBufferLength += 2;

	pMLMEContext->BulkOutSize = TransferBufferLength;
	//NdisReleaseSpinLock(&pAd->DataQLock[0]);
	RTUSB_SET_BULK_FLAG(pAd, fRTUSB_BULK_OUT_MLME);
	RTUSBKickBulkOut(pAd);
	//RTUSBWriteMACRegister(pAd, TX_CNTL_CSR, BIT8[QID_AC_BE]); // use AC_BE
}

/*
	==========================================================================
	Description:
		Send out a ACK frame to a specified STA upon receiving PS-POLL
	==========================================================================
 */
VOID ApEnqueueAckFrame(
	IN PRTMP_ADAPTER pAd,
	IN PUCHAR        pAddr,
	IN UCHAR         TxRate)
{
	NDIS_STATUS    NState;
	PHEADER_802_11  pAckFr;

	// since TxRate may change, we have to change Duration each time
	NState = MlmeAllocateMemory(pAd, (PVOID)&pAckFr);
	if (NState == NDIS_STATUS_SUCCESS)
	{
		MgtMacHeaderInit(pAd, pAckFr, SUBTYPE_ACK, 0, pAddr, pAd->CommonCfg.Bssid);
		pAckFr->FC.Type = BTYPE_CNTL;
		MiniportMMRequest(pAd, (PUCHAR)pAckFr, 10);
		MlmeFreeMemory(pAd, pAckFr);
	}
}

/*
	==========================================================================
	Description:
		This routine is called at initialization. It returns a channel number
		that complies to regulation domain and less interference with current
		enviornment.
	Return:
		ch -  channel number that
	NOTE:
		the retrun channel number is guaranteed to comply to current regulation
		domain that recorded in pAd->CommonCfg.CountryRegion
	==========================================================================
 */
UCHAR APAutoSelectChannel(
	IN PRTMP_ADAPTER pAd)
{
	UCHAR ch;
	UCHAR dirtyness[MAX_NUM_OF_CHANNELS+1], dirty;
	UCHAR max_rssi[MAX_NUM_OF_CHANNELS+1];
	ULONG FalseCca, FcsError;
//	BOOLEAN bFindIt = FALSE;

	// passive scan channel 1-14. collect statistics
	NdisZeroMemory(dirtyness, MAX_NUM_OF_CHANNELS+1);

	if (pAd->CommonCfg.PhyMode == PHY_11A)
	{
		if (pAd->CommonCfg.bIEEE80211H)
		{
			while(TRUE)
			{
				ch = pAd->ChannelList[RandomByte(pAd)%pAd->ChannelListNum].Channel;

				if (ch == 0)
					ch = FirstChannel(pAd);
				if (!RadarChannelCheck(pAd, ch))
					continue;

				/*for (i=0; i<pAd->ChannelListNum; i++)
				{
					if (pAd->ChannelList[i].Channel == ch)
					{
						if (pAd->ChannelList[i].RemainingTimeForUse == 0)
							bFindIt = TRUE;

						break;
					}
				}

				if (bFindIt == TRUE)
					break;*/
			};
		}
		else
		{
			ch = pAd->ChannelList[RandomByte(pAd)%pAd->ChannelListNum].Channel;
			if (ch == 0)
				ch = FirstChannel(pAd);
		}
		DBGPRINT(RT_DEBUG_TRACE,("1.APAutoSelectChannel pick up ch#%d\n",ch));
		return ch;
	}
	else
	{
		for (ch=14; ch>=1; ch--)
		{
			AsicSwitchChannel(pAd, ch);
			pAd->Counters8023.GoodReceives = 0;
			pAd->Counters8023.RxErrors = 0;
			pAd->ApCfg.AutoChannel_MaxRssi = 0;
			pAd->ApCfg.AutoChannel_Channel = ch;
			max_rssi[ch]=0;
			NdisMSleep(500000); // 0.5 sec at each channel
			max_rssi[ch] = pAd->ApCfg.AutoChannel_MaxRssi;

			RTUSBReadMACRegister(pAd, STA_CSR1, &FalseCca);
			FalseCca = FalseCca >> 16;
			RTUSBReadMACRegister(pAd, STA_CSR0, &FcsError);
			FcsError &= 0x0000ffff;

	//      if (pAd->Counters8023.GoodReceives)
			if (max_rssi[ch])
			{
				dirtyness[ch] += 10;
				if (ch > 1) dirtyness[ch-1] += 1;
				if (ch > 2) dirtyness[ch-2] += 1;
				if (ch > 3) dirtyness[ch-3] += 1;
				if (ch > 4) dirtyness[ch-4] += 1;
				if (ch < 14) dirtyness[ch+1] += 1;
				if (ch < 13) dirtyness[ch+2] += 1;
				if (ch < 12) dirtyness[ch+3] += 1;
				if (ch < 11) dirtyness[ch+4] += 1;
			}
			DBGPRINT(RT_DEBUG_TRACE,("Msleep at ch#%d to collect RX=%d, RSSI=%d, CRC error =%d, False CCA =%d\n",
				ch, pAd->Counters8023.GoodReceives, max_rssi[ch] - pAd->BbpRssiToDbmDelta, FcsError, FalseCca));
		}
		pAd->ApCfg.AutoChannel_Channel = 0;

		DBGPRINT(RT_DEBUG_TRACE,("Dirtyness = %d.%d.%d.%d - %d.%d.%d.%d - %d.%d.%d.%d - %d.%d\n",
			dirtyness[1], dirtyness[2], dirtyness[3], dirtyness[4], dirtyness[5], dirtyness[6], dirtyness[7],
			dirtyness[8], dirtyness[9], dirtyness[10], dirtyness[11], dirtyness[12], dirtyness[13], dirtyness[14]));

		// RULE 1. pick up a good channel that no one used
		for (ch = FirstChannel(pAd); ch !=0; ch = NextChannel(pAd, ch))
		{
			if (dirtyness[ch] == 0) break;
		}
		if (ch != 0)
		{
			DBGPRINT(RT_DEBUG_TRACE,("APAutoSelectChannel pick up ch#%d\n",ch));
			return ch;
		}

		// RULE 2. if not available, then co-use a channel that's no interference (dirtyness=10)
		// RULE 3. if not available, then co-use a channel that has minimum interference (dirtyness=11,12)
		for (dirty = 10; dirty <= 12; dirty++)
		{
			UCHAR candidate[MAX_NUM_OF_CHANNELS+1], candidate_num;
			UCHAR min_rssi = 255, final_channel = 0;

			candidate_num = 0;
			NdisZeroMemory(candidate, MAX_NUM_OF_CHANNELS+1);
			for (ch = FirstChannel(pAd); ch != 0; ch = NextChannel(pAd, ch))
			{
				if (dirtyness[ch] == dirty)
				{
					candidate[ch]=1;
					candidate_num++;
				}
			}
			// if there's more than 1 candidate, pick up the channel with minimum RSSI
			if (candidate_num)
			{
				for (ch = 1; ch <= 14; ch ++)
				{
					if (candidate[ch] && (max_rssi[ch] < min_rssi))
					{
						final_channel = ch;
						min_rssi = max_rssi[ch];
					}
				}
				if (final_channel != 0)
				{
					DBGPRINT(RT_DEBUG_TRACE,("APAutoSelectChannel pick up ch#%d\n",final_channel));
					return final_channel;
				}
			}
		}

		// RULE 4. still not available, pick up the first channel
		ch = FirstChannel(pAd);
		DBGPRINT(RT_DEBUG_TRACE,("APAutoSelectChannel pick up ch#%d\n",ch));
		return ch;
	}
}

