/**************
 *
 * Filename:    connectmgr.c
 *
 * Purpose:     Connection Manager application
 *
 * Copyright: © 2017 Sierra Wireless Inc., all rights reserved
 *
 **************/
#define __STDC_FORMAT_MACROS
#include <pthread.h>
#include <inttypes.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <time.h>
#include <fcntl.h>
#include <sys/msg.h>
#include <sys/wait.h>
#include <string.h>
#include <errno.h>

#define VERSION	"1.0.2111.0"

//
// Compile Switch definitions for different behavior of the program
//

#define QMUX_ROUTER_SUPPORT 1

// Whether to show invalid profile error.  If set to 1, querying empty profiles will show error
#define SHOW_INVALID_PROFILE_ERROR 0

// When making IPv4v6 connection, connect to IPv6 first or IPv4 first
// Set to connect to IPv6 first as a workaround to HSWOEMP-1595
#define FIRST_CONNECTION_IPV6	1

//
// End of Compile Switches
//

// Using MBIM to set QMAP
#include <MbimTransport.h>
#include <BasicConnectDeviceService.h>
#include <MbimSyncObject.h>
#include <MbimTransaction.h>

#include "displaysettings.h"
#include "msgid.h"
#include "dms.h"
#include "nas.h"
#include "wds.h"
#include "tmd.h"
#include "ts.h"
#include "qos.h"

// Using WDA to set QMAP
#include "wda.h"

#include "lite-qmi-demo.h"
#include <QmuxTransport.h>
#include <QmiService.h>
#include <CtlService.h>
#include <QmiSyncObject.h>

#include "ping.h"
#include "netlink_util.h"
#include "dev_util.h"
#include "qmux_util.h"
#include "proc_util.h"
#include "str_util.h"

/****************************************************************
*                       DEFINES
****************************************************************/

#define SUCCESS                  0
#define ENTER_KEY                0x0A
#define ENTER_KEY_PRESSED        0
#define OPTION_LEN               5
#define IP_ADDRESS_LEN           15
#define IPv4_FAMILY_PREFERENCE   0x04
#define IPv6_FAMILY_PREFERENCE   0x06
#define IPv4v6_FAMILY_PREFERENCE 0x07
#define MAX_FIELD_SIZE           128
#define MAX_MASK_SIZE			 20
#define MAX_INT16_VALUE_SIZE     0x7FFF
#define MAX_INT32_VALUE_SIZE	 0x7FFFFFFF
#define MAX_UINT32_VALUE_SIZE	 0xFFFFFFFF
#define MAX_UINT64_VALUE_SIZE	 0xFFFFFFFFFFFFFFFF

/* APN, User Name and Profile name size should be 3 greater than the actual
 * characters to be read. This will help to ensure that user should not enter
 * more than maximum allowed characters.
 */
#define MAX_APN_SIZE            104
#define MAX_PROFILE_NAME_SIZE   17
#define MAX_USER_NAME_SIZE      129
#define MIN_PROFILES            1
#define MAX_PROFILES            42
#define PROFILE_TYPE_UMTS       0
#define TECHNOLOGY_3GPP         1
#define TECHNOLOGY_3GPP2        2

#define CDMA_PROFILE_OFFSET     (100)

#define IPV4 4
#define IPV6 6
#define MAX_QMAP_INSTANCE 8
#define MAX_IFLEN	64

 // Behavior upon connection establishment
#define PING_COUNT	4
#define PING_DELAY	3

// HSWOEMP-1595
#if FIRST_CONNECTION_IPV6
#define SPDN_CONN_START			1
#define SPDN_LIMIT				0
#define MPDN_CONN_START			3
#define MPDN_LIMIT				2
#else
#define SPDN_CONN_START			0
#define SPDN_LIMIT				2
#define MPDN_CONN_START			2
#define MPDN_LIMIT				4
#endif


int g_auto_test_enable = 1;
int g_mtu_auto_update_enable = 1;
int g_ip_auto_assign = 1;

const char* const g_enterActionLeaveEmpty = "leave the field as blank";
const char* const g_enterActionExit = "exit";
const char* const g_enterActionDefault = "use default value";
const char* const g_enterActionZero = "set value 0";

/****************************************************************
*                       DATA STRUCTURE
****************************************************************/

enum
{
	eLITE_CONNECT_APP_OK,
	eLITE_CONNECT_APP_ERR_QMI,
};

/* User options enumeration */
enum eUserOptions {
#ifdef ALL_CONNECTION_TYPES
	eSTART_UMTS_DATA_SESSION = 1,
	eSTART_LTE_DATA_SESSION,
	eSTART_CDMA_DATA_SESSION,
	eSTART_CDMA_RUIM_DATASESSION,
#else
	eSTART_LTE_DATA_SESSION = 1,
#endif
	eSTART_PDN_DATASESSION,
	eSTOP_DATA_SESSION,
	eSTOP_DATA_SESSIONS,
	eDISPLAY_ALL_PROFILES,
	eDISPLAY_SINGLE_PROFILE,
	eCREATE_PROFILE,
	eMODIFY_PROFILE_SETTINGS,
	eDELETE_PROFILE,
	eSCAN_NETWORKS,
	eEnable_QOS_EVENT,
	eDisable_QOS_EVENT,
	eREQUEST_QOSEX,
	eGET_QOSINFO,
	eSet_QOSIndicationRegister,
	eGET_PKT_STATS,
	eGET_CURRENT_CHANNEL_RATE,
	eRESET_DEVICE,
	eSET_POWER_MODE,
	eQOS_TESTING
};

enum eConnectionState {
	NONE,
	SINGLE,
	MULTIPDN
};

/* Profile indexes for profile existing on device */
struct profileIndexesInfo {
	uint8_t profileIndex[MAX_PROFILES];
	uint8_t totalProfilesOnDevice;
};

/* Profile Information structure */
struct profileInformation {
	uint8_t profileType;
	uint8_t PDPType;
	uint32_t IPAddress;
	uint32_t primaryDNS;
	uint32_t secondaryDNS;
	uint8_t Authentication;
	uint8_t  profileName[MAX_PROFILE_NAME_SIZE];
	uint8_t  APNName[MAX_APN_SIZE];
	uint8_t  userName[MAX_USER_NAME_SIZE];
	uint8_t  password[MAX_FIELD_SIZE];
};

/****************************************************************
*                    GLOBAL DATA
****************************************************************/
#define QMI_MSG_MAX 2048

const int nMaskPrefLenV4 = 0;//24;
const int nMaskPrefLenV6 = 124;

#define NUM_WDS_SVC	8

uint8_t g_modem_index = 0;

char szEthName[64];
const char* g_default_dev = NULL;

/* device connectivity */
static struct profileIndexesInfo indexInfo;

static CtlService s_CtlService;
static QmiService s_DmsService;
static QmiService s_NasService;
static QmiService s_TmdService;
static QmiService s_TsService;
static QmiService s_QosService;
static QmiService s_WdaService;
static QmuxTransport s_Transport;

static uint8_t g_NetSelPref = 0;
static uint16_t g_NasPendingXid = 0;

static unpack_tmd_SLQSTmdGetMitigationDevList_t tmdDevList;
static unpack_ts_SLQSTsGetSensorList_t tsSensorList;

bool g_exiting = false;

struct session
{
	QmiService wdsSvc;
	bool active;
	uint32_t id;
	char ip[INET6_ADDRSTRLEN];
	int prefixlen;
	uint8_t uIPCount;
} sessions[NUM_WDS_SVC];

#define RMNET_IF_ID_UNSET -1

int g_mode = QMUX_INTERFACE_UNKNOWN;
int g_rmnet_if_id = RMNET_IF_ID_UNSET;
bool g_vlan_added = false;
bool g_qmimux_added = false;
enum eConnectionState g_connection_state = NONE;
unsigned int g_nMtu = 0xFFFFFFFF;
unsigned int g_nMtuV0 = 0xFFFFFFFF;
unsigned int g_nMtuV1 = 0xFFFFFFFF;

struct routeTable
{
	char szif[MAX_IFLEN];
	char dst[INET6_ADDRSTRLEN];
	char gw[INET6_ADDRSTRLEN];
} setrouteTable[NUM_WDS_SVC][MAX_QMAP_INSTANCE] =
{
	{
		//{{""},{"8.8.8.0"},{""}},
		{{""},{"0.0.0.0"},{"0.0.0.0"}},
		{{""},{"216.105.38.0"},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}}
	},
	{
		//{{""},{"2607:f8b0:400a:800::2000"},{""}},
		{{""},{"2001:4860:4860::8888"},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}}
	},
	{
		//{{""},{"4.2.2.0"},{""}},
		{{""},{"0.0.0.0"},{"0.0.0.0"}},
		{{""},{"216.105.38.0"},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}}
	},
	{
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}},
		{{""},{""},{""}}
	}
};

char pingTable[NUM_WDS_SVC][MAX_QMAP_INSTANCE][INET6_ADDRSTRLEN] =
{
	{
	  {"8.8.8.8"},
	  {"216.105.38.13"},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""}
	},
	{
	  //{"2607:f8b0:400a:800::200e"},
	  {"2001:4860:4860::8888"},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""}
	},
	{
	  {"4.2.2.2"},
	  {"216.105.38.13"},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""}
	},
	{
	  {"fc01:cafe::1"},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""},
	  {""}
	}
};

/****************************************************************
*              FORWARD FUNCTION DECLARATION
****************************************************************/
static void display_device_info(void);
static void display_profile_info(uint8_t profileId, bool bDisplayHeader, bool bUpdateCache);
static void display_all_profiles(bool bShowNoProfile);


/****************************************************************
*                       FUNCTIONS
****************************************************************/

// Return message ID of the indication
static uint16_t IndicationCallback(const char* svcName, uint8_t svc, uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	(void)pIndicationCallbackContext;

	unpack_qmi_t rsp_ctx;
	dlog(eLOG_INFO, "\n<< receiving %s\n", helper_get_resp_ctx(svc, qmiPacket, qmiPacketSize, &rsp_ctx));
	dlog(eLOG_INFO, "%s IND: msgid 0x%x, type:%x\n", svcName, rsp_ctx.msgid, rsp_ctx.type);
	// TODO:

	return rsp_ctx.msgid;
}

static void DmsIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	if (IndicationCallback("DMS", eDMS, qmiPacket, qmiPacketSize, pIndicationCallbackContext) != 0x01)
		return;

	unpack_dms_SetEventReport_ind_t ind_rsp;
	if (unpack_dms_SetEventReport_ind(qmiPacket, qmiPacketSize, &ind_rsp) != SUCCESS)
		return;

	dlog(eLOG_INFO, "[unpack_dms_SetEventReport_ind]\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x10))	// Power state
		printf("\tPower state\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x11))	// PIN 1 status (deprecated)
		printf("\tPIN 1 status (deprecated)\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x12))	// PIN 2 status (deprecated)
		printf("\tPIN 2 status (deprecated)\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x13))	// Activation state
		printf("\tActivation state: %d\n", ind_rsp.ActivationStatusTlv.activationStatus);

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x14))	// Operating mode
		printf("\tOperating mode: %d\n", ind_rsp.OperatingModeTlv.operatingMode);

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x15))	// UIM state (deprecated)
		printf("\tUIM state (deprecated)\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x16))	// Wireless disable state
		printf("\tWireless disable state\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x17))	// PRL init notif
		printf("\tPRL init notif\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x18))	// CDMA lock mode state
		printf("\tCDMA lock mode state\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x19))	// Device multiSIM cap (dprecated)
		printf("\tDevice multiSIM cap (dprecated)\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x1A))	// Device multiSIM voice data cap
		printf("\tDevice multiSIM voice data cap\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x1B))	// Current subscription cap
		printf("\tCurrent subscription cap\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x1C))	// Subscription voice data cap
		printf("\tSubscription voice data cap\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x1D))	// Max active data subscriptions
		printf("\tMax active data subscriptions\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x1E))	// PRL info
		printf("\tPRL info\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x1F))	// Max device config list
		printf("\tMax device config list\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x20))	// Explicit configruation index
		printf("\tExplicit configruation index\n");
}

static void NasIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	IndicationCallback("NAS", eNAS, qmiPacket, qmiPacketSize, pIndicationCallbackContext);
}

static void TmdIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	if (g_exiting || IndicationCallback("TMD", eTMD, qmiPacket, qmiPacketSize, pIndicationCallbackContext) != 0x25)
		return;

	unpack_tmd_SLQSTmdMitigationLvlRptCallback_ind_t ind_rsp;
	if (unpack_tmd_SLQSTmdMitigationLvlRptCallback_ind(qmiPacket, qmiPacketSize, &ind_rsp) != SUCCESS)
		return;

	dlog(eLOG_INFO, "[unpack_tmd_SLQSTmdMitigationLvlRptCallback_ind]\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 1))
	{
		printf("\tdevice id len: %d\n", ind_rsp.deviceIdLen);
		printf("\tdevice id: %s\n", ind_rsp.deviceID);
	}

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 2))
		printf("\tlevel: %d\n", ind_rsp.lvl);
}

static void TsIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	if (g_exiting || IndicationCallback("TS", eTS, qmiPacket, qmiPacketSize, pIndicationCallbackContext) != 0x22)
		return;

	unpack_ts_SLQSTsTempRptCallback_ind_t ind_rsp;
	if (unpack_ts_SLQSTsTempRptCallback_ind(qmiPacket, qmiPacketSize, &ind_rsp) != SUCCESS)
		return;

	dlog(eLOG_INFO, "[unpack_ts_SLQSTsTempRptCallback_ind]\n");

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 1))
	{
		printf("\tsensor id len: %d\n", ind_rsp.sensor.sensorIdLen);
		printf("\tsensor id: %s\n", ind_rsp.sensor.sensorId);
	}

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 2))
		printf("\treport type: %d\n", ind_rsp.type);

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x10))
		printf("\ttemperature: %.1f degree C\n", ind_rsp.temp);

	if (swi_uint256_get_bit(ind_rsp.ParamPresenceMask, 0x11))
		printf("\tseq #: %d\n", ind_rsp.seqNum);
}

static void WdsIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	IndicationCallback("WDS", eWDS, qmiPacket, qmiPacketSize, pIndicationCallbackContext);
}

static void QosIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	uint16_t msgid = IndicationCallback("QOS", eQOS, qmiPacket, qmiPacketSize, pIndicationCallbackContext);
	int unpackRetCode = 0;

	switch (msgid)
    {
        case eQMI_QOS_EVENT_IND:
		{
			unpack_qos_SLQSSetQosEventCallback_ind_t setQosEventCallback;
			unpack_qos_EventCallback_t qosEventCallback;			
			uint16_t profile_value[10] = {0};
    		qosEventCallback.Profile_count = 10;
    		qosEventCallback.pProfile_value = profile_value;
			setQosEventCallback.pNetworkSupportedQoSProfiles = &qosEventCallback;

			unpackRetCode = unpack_qos_SLQSSetQosEventCallback_ind(qmiPacket, qmiPacketSize, &setQosEventCallback);
            if (!unpackRetCode)
            {
                printf("unpack_qos_SLQSSetQosEventCallback_ind failed errorCode: %d - %s\n", unpackRetCode, helper_get_error_reason(unpackRetCode));
            }
            else
            {
				if (swi_uint256_get_bit (setQosEventCallback.ParamPresenceMask, 0x11))
				{
                	printf("Technology preference: %d\n", qosEventCallback.Tech_pref);
                	printf("Number of profile values: %d\n", qosEventCallback.Profile_count);
					for (int idx = 0; idx < qosEventCallback.Profile_count; ++idx)
                		printf("profile values: %d\n", qosEventCallback.pProfile_value[idx]);
				}
            }
        }
        break;
		case eQMI_QOS_GLOBAL_QOS_FLOW_IND:
		{
			unpack_qos_QmiCbkQosFlowStatus_t	sQosFlowStatus;
			memset(&sQosFlowStatus, 0, sizeof(unpack_qos_QmiCbkQosFlowStatus_t));
			unpack_qos_QmiCbkQosFlowGranted_t	sTxQosFlowGranted;
			memset(&sTxQosFlowGranted, 0, sizeof(unpack_qos_QmiCbkQosFlowGranted_t));
			unpack_qos_QmiCbkQosFlowGranted_t	sRxQosFlowGranted;
			memset(&sRxQosFlowGranted, 0, sizeof(unpack_qos_QmiCbkQosFlowGranted_t));
			unpack_qos_QmiCbkQosFilters_t		sTxQosFilters;
			memset(&sTxQosFilters, 0, sizeof(unpack_qos_QmiCbkQosFilters_t));
			unpack_qos_QmiCbkQosFilters_t		sRxQosFilters;
			memset(&sRxQosFilters, 0, sizeof(unpack_qos_QmiCbkQosFilters_t));
			uint32_t							nFlow_type = 0;
			uint8_t								nBearer_id = 0;
			uint16_t							nFc_seq_num = 0;
			uint32_t							nTx_5g_qci = 0;
			uint32_t							nRx_5g_qci = 0;
			uint16_t							nTx_averaging_window = 0;
			uint16_t							nRx_averaging_window = 0;
			unpack_qos_QmiCbkQosTxFilterMatchAll_t sTxFilterMatchAll;
			memset(&sTxFilterMatchAll, 0, sizeof(unpack_qos_QmiCbkQosTxFilterMatchAll_t));

			unpack_qos_SLQSQosGlobalQosFlow_ind_t sQosGlobalQosFlowInd;
			sQosGlobalQosFlowInd.pQosFlowStatus = &sQosFlowStatus;
			sQosGlobalQosFlowInd.pTxQosFlowGranted = &sTxQosFlowGranted;
			sQosGlobalQosFlowInd.pRxQosFlowGranted = &sRxQosFlowGranted;
			sQosGlobalQosFlowInd.pTxQosFilters = &sTxQosFilters;
			sQosGlobalQosFlowInd.pRxQosFilters = &sRxQosFilters;
			sQosGlobalQosFlowInd.pFlow_type = &nFlow_type;
			sQosGlobalQosFlowInd.pBearer_id = &nBearer_id;
			sQosGlobalQosFlowInd.pFc_seq_num = &nFc_seq_num;
			sQosGlobalQosFlowInd.pTx_5g_qci = &nTx_5g_qci;
			sQosGlobalQosFlowInd.pRx_5g_qci = &nRx_5g_qci;
			sQosGlobalQosFlowInd.pTx_averaging_window = &nTx_averaging_window;
			sQosGlobalQosFlowInd.pRx_averaging_window = &nRx_averaging_window;
			sQosGlobalQosFlowInd.pTxFilterMatchAll = &sTxFilterMatchAll;

			unpackRetCode = unpack_qos_SLQSQosGlobalQosFlow_ind(qmiPacket, qmiPacketSize, &sQosGlobalQosFlowInd);
			if (!unpackRetCode)
			{
				printf("unpack_qos_SLQSQosGlobalQosFlow_ind failed errorCode: %d - %s\n", unpackRetCode, helper_get_error_reason(unpackRetCode));
			}
			else
			{
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x01))
				{
					printf("\nQoS Flow State:\n");
					printf("QoS identifier= %d\n", sQosFlowStatus.qos_id);
					printf("new flow= %d\n", sQosFlowStatus.new_flow);
					printf("Flow state change flow= %d\n", sQosFlowStatus.state_change);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x10))
				{
					printf("\nTx QoS Flow Granted:\n");
					printf("flow_valid_params= %"PRIu64"\n", sTxQosFlowGranted.flow_valid_params);
					printf("ip_flow_trf_cls= %d\n", sTxQosFlowGranted.ip_flow_trf_cls);
					printf("data_rate_max= %"PRIu64"\n", sTxQosFlowGranted.data_rate_max);
					printf("guaranteed_rates= %"PRIu64"\n", sTxQosFlowGranted.guaranteed_rate);
					printf("peak_rate= %d\n", sTxQosFlowGranted.peak_rate);
					printf("token_rate= %d\n", sTxQosFlowGranted.token_rate);
					printf("bucket_size= %d\n", sTxQosFlowGranted.bucket_size);
					printf("ip_flow_latency= %d\n", sTxQosFlowGranted.ip_flow_latency);
					printf("ip_flow_pkt_error_rate_multiplier= %d\n", sTxQosFlowGranted.ip_flow_pkt_error_rate_multiplier);
					printf("ip_flow_pkt_error_rate_exponent= %d\n", sTxQosFlowGranted.ip_flow_pkt_error_rate_exponent);
					printf("ip_flow_min_policed_packet_size= %d\n", sTxQosFlowGranted.ip_flow_min_policed_packet_size);
					printf("ip_flow_max_allowed_packet_size= %d\n", sTxQosFlowGranted.ip_flow_max_allowed_packet_size);
					printf("ip_flow_3gpp_residual_bit_error_rate= %d\n", sTxQosFlowGranted.ip_flow_3gpp_residual_bit_error_rate);
					printf("ip_flow_3gpp_traffic_handling_priority= %d\n", sTxQosFlowGranted.ip_flow_3gpp_traffic_handling_priority);
					printf("ip_flow_3gpp2_profile_id= %d\n", sTxQosFlowGranted.ip_flow_3gpp2_profile_id);
					printf("ip_flow_3gpp2_flow_priority= %d\n", sTxQosFlowGranted.ip_flow_3gpp2_flow_priority);
					printf("ip_flow_3gpp_im_cn_flag= %d\n", sTxQosFlowGranted.ip_flow_3gpp_im_cn_flag);
					printf(" ip_flow_3gpp_sig_ind= %d\n", sTxQosFlowGranted.ip_flow_3gpp_sig_ind);
					printf("ip_flow_lte_qci= %d\n", sTxQosFlowGranted.ip_flow_lte_qci);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x11))
				{
					printf("\nRx QoS Flow Granted:\n");
					printf("flow_valid_params= %"PRIu64"\n", sRxQosFlowGranted.flow_valid_params);
					printf("ip_flow_trf_cls= %d\n", sRxQosFlowGranted.ip_flow_trf_cls);
					printf("data_rate_max= %"PRIu64"\n", sRxQosFlowGranted.data_rate_max);
					printf("guaranteed_rates= %"PRIu64"\n", sTxQosFlowGranted.guaranteed_rate);
					printf("peak_rate= %d\n", sRxQosFlowGranted.peak_rate);
					printf("token_rate= %d\n", sRxQosFlowGranted.token_rate);
					printf("bucket_size= %d\n", sRxQosFlowGranted.bucket_size);
					printf("ip_flow_latency= %d\n", sRxQosFlowGranted.ip_flow_latency);
					printf("ip_flow_pkt_error_rate_multiplier= %d\n", sRxQosFlowGranted.ip_flow_pkt_error_rate_multiplier);
					printf("ip_flow_pkt_error_rate_exponent= %d\n", sRxQosFlowGranted.ip_flow_pkt_error_rate_exponent);
					printf("ip_flow_min_policed_packet_size= %d\n", sRxQosFlowGranted.ip_flow_min_policed_packet_size);
					printf("ip_flow_max_allowed_packet_size= %d\n", sRxQosFlowGranted.ip_flow_max_allowed_packet_size);
					printf("ip_flow_3gpp_residual_bit_error_rate= %d\n", sRxQosFlowGranted.ip_flow_3gpp_residual_bit_error_rate);
					printf("ip_flow_3gpp_traffic_handling_priority= %d\n", sRxQosFlowGranted.ip_flow_3gpp_traffic_handling_priority);
					printf("ip_flow_3gpp2_profile_id= %d\n", sRxQosFlowGranted.ip_flow_3gpp2_profile_id);
					printf("ip_flow_3gpp2_flow_priority= %d\n", sRxQosFlowGranted.ip_flow_3gpp2_flow_priority);
					printf("ip_flow_3gpp_im_cn_flag= %d\n", sRxQosFlowGranted.ip_flow_3gpp_im_cn_flag);
					printf("ip_flow_3gpp_sig_ind= %d\n", sRxQosFlowGranted.ip_flow_3gpp_sig_ind);
					printf("ip_flow_lte_qci= %d\n", sRxQosFlowGranted.ip_flow_lte_qci);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x12))
				{
					printf("\nRx QoS Filters:\n");

					printf("tx_qos_filters_len= %d\n", sRxQosFilters.tx_qos_filters_len);
					printf("ip_version= %d\n", sRxQosFilters.ip_version);
					printf("valid_params_ipv4= %"PRIu64"\n", sRxQosFilters.valid_params_ipv4);
					if (sRxQosFilters.valid_params_ipv4)
					{
						printf("ipv4_addr_1= %d\n", sRxQosFilters.ipv4_addr_1);
						printf("subnet_mask_1= %d\n", sRxQosFilters.subnet_mask_1);
						printf("ipv4_addr_2= %d\n", sRxQosFilters.ipv4_addr_2);
						printf("subnet_mask_2= %d\n", sRxQosFilters.subnet_mask_2);
						printf("val_ipv4= %d\n", sRxQosFilters.val_ipv4);
						printf("mask_ipv4 = %d\n", sRxQosFilters.mask_ipv4);
					}
					printf("valid_params_ipv6= %"PRIu64"\n", sRxQosFilters.valid_params_ipv6);
					if (sRxQosFilters.valid_params_ipv6)
					{
						printf("prefix_len_1= %d\n", sRxQosFilters.prefix_len_1);
						printf("IPv6: ");
						for (int idx = 0; idx < sRxQosFilters.prefix_len_1; ++idx)
							printf("%d:", sRxQosFilters.ipv6_address_1[idx]);

						printf("\nprefix_len_2= %d\n", sRxQosFilters.prefix_len_2);
						printf("IPv6: ");
						for (int idx = 0; idx < sRxQosFilters.prefix_len_2; ++idx)
							printf("%d:", sRxQosFilters.ipv6_address_2[idx]);

						printf("\nval_ipv6 = %d\n", sRxQosFilters.val_ipv6);
						printf("mask_ipv6 = %d\n", sRxQosFilters.mask_ipv6);
						printf("flow_label_ipv6 = %d\n", sRxQosFilters.flow_label_ipv6);
						printf("xport_protocol = %d\n", sRxQosFilters.xport_protocol);
					}
					printf("valid_params_port_1 = %"PRIu64"\n", sRxQosFilters.valid_params_port_1);
					if (sRxQosFilters.valid_params_port_1)
					{
						printf("port_1 = %d\n", sRxQosFilters.port_1);
						printf("range_1 = %d\n", sRxQosFilters.range_1);
						printf("port_2 = %d\n", sRxQosFilters.port_2);
						printf("range_2 = %d\n", sRxQosFilters.range_2);
					}
					printf("valid_params_port_2 = %"PRIu64"\n", sRxQosFilters.valid_params_port_2);
					if (sRxQosFilters.valid_params_port_2)
					{
						printf("port_3 = %d\n", sRxQosFilters.port_3);
						printf("range_3 = %d\n", sRxQosFilters.range_3);
						printf("port_4 = %d\n", sRxQosFilters.port_4);
						printf("range_4 = %d\n", sRxQosFilters.range_4);
					}
					printf("valid_params_icmp = %"PRIu64"\n", sRxQosFilters.valid_params_icmp);
					if (sRxQosFilters.valid_params_icmp)
					{
						printf("type_icmp = %d\n", sRxQosFilters.type_icmp);
						printf("code_icmp = %d\n", sRxQosFilters.code_icmp);
					}
					printf("valid_params_spi_1 = %"PRIu64"\n", sRxQosFilters.valid_params_spi_1);
					if(sRxQosFilters.valid_params_spi_1)
						printf("spi_1 = %d\n", sRxQosFilters.spi_1);
					printf("valid_params_spi_2 = %"PRIu64"\n", sRxQosFilters.valid_params_spi_2);
					if (sRxQosFilters.valid_params_spi_2)
						printf("spi_2= %d\n", sRxQosFilters.spi_2);
					printf("filter_id = %d\n", sRxQosFilters.filter_id);
					printf("filter_precedence= %d\n", sRxQosFilters.filter_precedence);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x13))
				{
					printf("\nRx QoS Filters:\n");

					printf("tx_qos_filters_len= %d\n", sRxQosFilters.tx_qos_filters_len);
					printf("ip_version= %d\n", sRxQosFilters.ip_version);
					printf("valid_params_ipv4= %"PRIu64"\n", sRxQosFilters.valid_params_ipv4);
					if (sRxQosFilters.valid_params_ipv4)
					{
						printf("ipv4_addr_1= %d\n", sRxQosFilters.ipv4_addr_1);
						printf("subnet_mask_1= %d\n", sRxQosFilters.subnet_mask_1);
						printf("ipv4_addr_2= %d\n", sRxQosFilters.ipv4_addr_2);
						printf("subnet_mask_2= %d\n", sRxQosFilters.subnet_mask_2);
						printf("val_ipv4= %d\n", sRxQosFilters.val_ipv4);
						printf("mask_ipv4 = %d\n", sRxQosFilters.mask_ipv4);
					}
					printf("valid_params_ipv6 = %"PRIu64"\n", sRxQosFilters.valid_params_ipv6);
					if (sRxQosFilters.valid_params_ipv6)
					{
						printf("prefix_len_1 = %d\n", sRxQosFilters.prefix_len_1);
						printf("IPv6: ");
						for (int idx = 0; idx < sRxQosFilters.prefix_len_1; ++idx)
							printf("%d:", sRxQosFilters.ipv6_address_1[idx]);

						printf("\nprefix_len_2 = %d\n", sRxQosFilters.prefix_len_2);
						printf("IPv6: ");
						for (int idx = 0; idx < sRxQosFilters.prefix_len_2; ++idx)
							printf("%d:", sRxQosFilters.ipv6_address_2[idx]);

						printf("\nval_ipv6 = %d\n", sRxQosFilters.val_ipv6);
						printf("mask_ipv6 = %d\n", sRxQosFilters.mask_ipv6);
						printf("flow_label_ipv6 = %d\n", sRxQosFilters.flow_label_ipv6);
						printf("xport_protocol = %d\n", sRxQosFilters.xport_protocol);
					}
					printf("valid_params_port_1 = %"PRIu64"\n", sRxQosFilters.valid_params_port_1);
					if (sRxQosFilters.valid_params_port_1)
					{
						printf("port_1 = %d\n", sRxQosFilters.port_1);
						printf("range_1 = %d\n", sRxQosFilters.range_1);
						printf("port_2 = %d\n", sRxQosFilters.port_2);
						printf("range_2 = %d\n", sRxQosFilters.range_2);
					}
					printf("valid_params_port_2 = %"PRIu64"\n", sRxQosFilters.valid_params_port_2);
					if (sRxQosFilters.valid_params_port_2)
					{
						printf("port_3 = %d\n", sRxQosFilters.port_3);
						printf("range_3 = %d\n", sRxQosFilters.range_3);
						printf("port_4 = %d\n", sRxQosFilters.port_4);
						printf("range_4 = %d\n", sRxQosFilters.range_4);
					}
					printf("valid_params_icmp = %"PRIu64"\n", sRxQosFilters.valid_params_icmp);
					if (sRxQosFilters.valid_params_icmp)
					{
						printf("type_icmp = %d\n", sRxQosFilters.type_icmp);
						printf("code_icmp = %d\n", sRxQosFilters.code_icmp);
					}
					printf("valid_params_spi_1 = %"PRIu64"\n", sRxQosFilters.valid_params_spi_1);
					if (sRxQosFilters.valid_params_spi_1)
						printf("spi_1 = %d\n", sRxQosFilters.spi_1);
					printf("valid_params_spi_2 = %"PRIu64"\n", sRxQosFilters.valid_params_spi_2);
					if (sRxQosFilters.valid_params_spi_2)
						printf("spi_2= %d\n", sRxQosFilters.spi_2);
					printf("filter_id = %d\n", sRxQosFilters.filter_id);
					printf("filter_precedence = %d\n", sRxQosFilters.filter_precedence);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x14))
				{
					printf("Flow_type= %d\n", nFlow_type);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x15))
				{
					printf("Bearer_id= %d\n", nBearer_id);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x16))
				{
					printf("Fc_seq_num= %d\n", nFc_seq_num);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x17))
				{
					printf("Tx_5g_qci= %d\n", nTx_5g_qci);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x18))
				{
					printf("Rx_5g_qci= %d\n", nRx_5g_qci);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x19))
				{
					printf("Tx_averaging_window= %d\n", nTx_averaging_window);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x1A))
				{
					printf("Rx_averaging_window= %d\n", nRx_averaging_window);
				}
				if (swi_uint256_get_bit(sQosGlobalQosFlowInd.ParamPresenceMask, 0x1B))
				{
					printf("Number of filter ID= %d\n", sTxFilterMatchAll.tx_filter_match_all_len);
					for (int idx = 0; idx < sTxFilterMatchAll.tx_filter_match_all_len; ++idx)
						printf("filter ID[%d]= %d\n", idx, sTxFilterMatchAll.filter_id[idx]);
				}
			}
		}
		break;
    }
}

/****************************************************************
*                    COMMON FUNCTIONS
****************************************************************/
/*
 * Name:     FlushStdinStream
 *
 * Purpose:  Flush the stdin stream
 *
 * Params:   None
 *
 * Return:   None
 *
 * Notes:    fflush does not work for input stream.
 */
void FlushStdinStream()
{
	int inputChar;

	/* keep on reading until a <New Line> or end of file is received */
	do
	{
		inputChar = getchar();

#ifdef DBG
		fprintf(stderr, "inputChar: 0x%x\n", inputChar);
#endif
	} while ((inputChar != ENTER_KEY) && (inputChar != EOF));
}

/*
 * Name:     GetStringFromUser
 *
 * Purpose:  Prompt the user to enter a string and store it in the received
 *           buffer.
 *
 * Params:   pFieldName - Name of the string to be retrieved.
 *           pBuffer    - Buffer to receive the string from user.
 *           bufLen     - Length of the buffer to receive the string from user.
 *
 * Return:   SUCCESS   - If the user enters a valid value
 *           ENTER_KEY - If the user presses the Enter key
 *
 * Notes:    None
 */
uint8_t GetStringFromUser(const char* pOptionList, const char* pFieldName, const char* pEnterAction, char* pBuffer, int bufLen)
{
	/* Print the menu */
	if (pOptionList)
		fprintf(stderr, "\n%s", pOptionList);

	fprintf(stderr, "\nPlease enter %s (up to %d Characters), or press <Enter> to %s: ",
		pFieldName, bufLen - 1, pEnterAction);

	/* Clear the buffer */
	memset(pBuffer, 0, bufLen);

	/* Receive the input from the user */
	while(fgets(pBuffer, bufLen, stdin) == NULL);

	/* If '/n' character is not read, there are more characters in input
		* stream. Clear the input stream.
		*/
	if (NULL == strchr(pBuffer, ENTER_KEY))
		FlushStdinStream();

	/* If only <ENTER> is pressed by the user, return to main menu */
	uint8_t rtn = ENTER_KEY == pBuffer[0] ? ENTER_KEY : SUCCESS;

	/* Remove the enter key read at the end of the buffer */
	size_t len = strlen(pBuffer);
	if (len > 0 && pBuffer[len - 1] == ENTER_KEY)
		pBuffer[len - 1] = '\0';

#ifdef DBG
	fprintf(stderr, "Entered Value : %s\n", pBuffer);
#endif

	return rtn;
}

/*
 * Name:     GetIPFromUser
 *
 * Purpose:  Prompt the user to enter the IP address and copy it in the passed
 *           buffer.
 *
 * Return:   SUCCESS   - In case valid IP Address is entered by the user
 *           ENTER_KEY - If enter key is pressed by the user
 *
 * Params:   pAddressString - Name of the address to be retrieved.
 *           pIPAddress     - Buffer to receive the address from user.
 *           pAddress       - Pointer to store received IP address after
 *                            conversion from dot notation to ULONG value.
 *
 * Notes:    None
 */
uint8_t GetIPFromUser(char *pAddressString, char *pIPAddress, uint32_t *pAddress)
{
	char szFieldName[MAX_FIELD_SIZE];
	sprintf(szFieldName, "%s Address in xxx.xxx.xxx.xxx format", pAddressString);

	do
	{
		if (GetStringFromUser(NULL, szFieldName, g_enterActionLeaveEmpty, pIPAddress, MAX_FIELD_SIZE) == ENTER_KEY)
		{
			*pAddress = 0;
			return ENTER_KEY;
		}

		uint8_t count = 0, IPAddressWrong = FALSE;

		/* Check if there is nothing followed by a Dot in the IP address or
		 * there are two adjacent dots.
		 */
		char *pCharacterOccurence = strchr(pIPAddress, '.');
		while (NULL != pCharacterOccurence)
		{
			if (('.' == pCharacterOccurence[1]) ||
				('\0' == pCharacterOccurence[1]))
			{
#ifdef DBG
				fprintf(stderr, "Two Adjacent dots or NULL after a dot:"\
					"Wrong IP Address\n");
#endif
				IPAddressWrong = TRUE;
				break;
			}

			count++;
			pCharacterOccurence = strchr((pCharacterOccurence + 1), '.');
		}

		/* If there are more than three dots in the IP address */
		if ((count != 3) || (IPAddressWrong == TRUE))
		{
#ifdef DBG
			fprintf(stderr, "Incorrect No. of dots in address : %d\n", count);
#endif
			IPAddressWrong = FALSE;
			count = 0;
			continue;
		}

		count = 0;

	/* Convert the IP address from DOT notation to ULONG */
	} while (inet_pton(AF_INET, pIPAddress, pAddress) == 0);

	/* Valid IP Address has been retrieved */
	return SUCCESS;
}

/*
 * Name:     GetIPv6FromUser
 *
 * Purpose:  Prompt the user to enter the IPV6 address and copy it in the passed
 *           buffer.
 *
 * Return:   SUCCESS   - In case valid IP Address is entered by the user
 *           ENTER_KEY - If enter key is pressed by the user
 *
 * Params:   pIPAddress     - Buffer to receive the address from user.
 *           pIpv6_address  - Pointer to store received IP address after
 *                            conversion from dot notation to uint8_t[16].
 * Notes:    None
 */
uint8_t GetIPv6FromUser(char *pIPAddress, uint8_t *pIpv6_address, uint8_t *pByteCount)
{
	char szFieldName[MAX_FIELD_SIZE] = {0};
	strcpy(szFieldName, "IPV6 Address in xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx format");

	if (GetStringFromUser(NULL, szFieldName, g_enterActionLeaveEmpty, pIPAddress, MAX_FIELD_SIZE) == ENTER_KEY)
	{
		*pIpv6_address = 0;
		*pIpv6_address = 0;

		return ENTER_KEY;
	}

	uint8_t index = 0;
	char *pSemicolon1 = pIPAddress;
	char szWord[5] = {0};
	*pByteCount = 0;
	uint8_t maxByte = 0xFF;
	int length = strlen(pIPAddress);

	char *pSemicolon2 = strchr(pIPAddress, ':');
	if (pSemicolon2 == NULL)
		pSemicolon2 = pIPAddress + length;

	uint16_t ipValue = 0;
	while ((pSemicolon2 - pSemicolon1) > 0 )
	{
		if (StrNCpy(szWord, pSemicolon1, pSemicolon2 - pSemicolon1) < (pSemicolon2 - pSemicolon1))
		{
#ifdef DBG
			fprintf(stderr, "More than 4 characters between two semicolons:"\
				"Wrong IP Address\n");
#endif
			return maxByte;
		}
		sscanf( szWord, "%"SCNx16, &ipValue);
		*pIpv6_address++ = ipValue >> 8;
		*pIpv6_address++ = (uint8_t)(ipValue & 0x00FF);
		index++;

		*pByteCount += 2;
		/* If there are more than 15 dots in the IP address */
		if (index > 8)
		{
#ifdef DBG
			fprintf(stderr, "Incorrect No. of semicolon in address : %d\n", count);
#endif
			return maxByte;
		}

		if (pSemicolon2 == (pIPAddress + length))
			break;
		pSemicolon1 = pSemicolon2 + 1;
		pSemicolon2 = strchr(pSemicolon1, ':');
		if (pSemicolon2 == NULL)
			pSemicolon2 = pIPAddress + length;
	}

	/* Valid IPV6 Address has been retrieved */
	return SUCCESS;
}

int GetDigits(int n)
{
	int nDigits = 0;
	do
	{
		nDigits++;
		n /= 10;
	} while (n != 0);
	
	return nDigits;
}

int GetNumericValue(const char* pOptionList, const char* pFieldName, const char* pEnterAction, int enterVal, int min, int max)
{
	int nValue = 0;

	do
	{
		char pBuf[OPTION_LEN];
		int nBufLen = GetDigits(max)+1;
		if (GetStringFromUser(pOptionList, pFieldName, pEnterAction, pBuf, nBufLen > OPTION_LEN ? OPTION_LEN : nBufLen) == ENTER_KEY)
			return enterVal;

		/* Convert the authentication value provided by user into integer */
		nValue = atoi(pBuf);
	} while (nValue < min || nValue > max);

	return nValue;
}

uint64_t GetDigits64(uint64_t n)
{
	uint64_t nDigits = 0;
	do
	{
		nDigits++;
		n /= 10;
	} while (n != 0);
	
	return nDigits;
}

uint64_t GetNumericValue64(const char* pOptionList, const char* pFieldName, const char* pEnterAction, uint enterVal, uint min, uint64_t max)
{
	uint64_t nValue = 0;

	do
	{
		char pBuf[MAX_MASK_SIZE];
		uint64_t nBufLen = GetDigits64(max)+1;
		if (GetStringFromUser(pOptionList, pFieldName, pEnterAction, pBuf, nBufLen > MAX_MASK_SIZE ? MAX_MASK_SIZE : nBufLen) == ENTER_KEY)
			return enterVal;

		/* Convert the authentication value provided by user into integer */
		nValue = (uint64_t)strtol(pBuf, NULL, 0);
	} while (nValue < min || nValue > max);

	return nValue;
}

/*
 * Name:     GetPDPType
 *
 * Purpose:  Prompt the user to enter the PDP Type.
 *
 * Params:   None
 *
 * Return:   PDP Type value provided by the user(between 0 - 3),
 *           or hex value of enter key if pressed by the user.
 *
 * Notes:    None
 */
int GetPDPType()
{
	return GetNumericValue(NULL, "PDP Type value( 0 - IPv4, 1 - PPP, 2 - IPV6, 3 - IPV4V6 )",
		g_enterActionExit, ENTER_KEY, 0, 3);
}

/*
 * Name:     GetAuthenticationValue
 *
 * Purpose:  Prompt the user to enter the Authentication value.
 *
 * Params:   None
 *
 * Return:   Authentication value provided by the user(between 0 - 3),
 *           or hex value of enter key if pressed by the user.
 *
 * Notes:    None
 */
int GetAuthenticationValue()
{
	return GetNumericValue(NULL, "Authentication value( 0 - None (default), 1 - PAP, 2 - CHAP, 3 - PAP/CHAP )",
		g_enterActionDefault, 0, 0, 3);
}

/*
 * Name:     GetIPFamilyPreference
 *
 * Purpose:  Prompt the user to enter the IP Family preference
 *
 * Params:   None
 *
 * Return:   IPFamilyPreference
 *
 * Notes:    None
 */
int GetIPFamilyPreference()
{
	switch (GetNumericValue("1: IPV4 (default)\n2: IPV6\n3: IPV4V6",
		"IP family preference for the call", g_enterActionDefault, 1, 1, 3))
	{
	case 2:
		return IPv6_FAMILY_PREFERENCE;
	case 3:
		return IPv4v6_FAMILY_PREFERENCE;
	//case 1:
	default:
		break;
	}

	return IPv4_FAMILY_PREFERENCE;
}

/*
 * Name:     GetUserProfileId
 *
 * Purpose:  Prompt the user to enter the profile id whose information needs to
 *           be retrieved.
 *
 * Params:   None
 *
 * Return:   Profile index selected by the user(between 1 - 42),
 *           or hex value of enter key if pressed by the user else FALSE.
 *
 * Notes:    None
 */
int GetUserProfileId()
{
	return GetNumericValue(NULL, "a profile id (1-42)", g_enterActionExit, ENTER_KEY_PRESSED, MIN_PROFILES, MAX_PROFILES);
}

void GetUserNetworkSelectionPreference(struct nas_netSelectionPref* pPref)
{
	pPref->netReg = GetNumericValue(NULL, "Network selection preference (0: Auto (default), 1: Manual)", g_enterActionDefault, 0, 0, 1);
	if (pPref->netReg == 1)
	{
		// Manual
		pPref->mcc = GetNumericValue(NULL, "3 digit MCC. If empty, use auto network selection", g_enterActionLeaveEmpty, 0, 100, 999);
		if (pPref->mcc != 0)
			pPref->mnc = GetNumericValue(NULL, "2 or 3 digit MNC. If empty, use auto network selection", g_enterActionLeaveEmpty, 0, 10, 999);

		if (pPref->mcc == 0 || pPref->mnc == 0)
		{
			pPref->netReg = 0;
			pPref->mcc = 0;
			pPref->mnc = 0;
		}
	}
}

int GetDataSession()
{
	for (int i = 0; i < NUM_WDS_SVC; i++)
	{
		printf("Session %d: %s\n", i, sessions[i].active ? "Active" : "Inactive");
	}
	int session = GetNumericValue("Select an active session to disconnect", "session", g_enterActionExit, ENTER_KEY, 0, NUM_WDS_SVC - 1);

	if (session == ENTER_KEY)
	{
		return -1;
	}

	if (!sessions[session].active)
	{
		dlog(eLOG_INFO, "Selected data session is not active.\n\n");
		return -1;
	}

	return session;
}

const char* GetMpdnIfName(int nIdx)
{
	if (g_vlan_added)
	{
		switch (nIdx)
		{
		case 0:
		case 1: return VLAN_0_NAME;
		case 2:
		case 3: return VLAN_1_NAME;
		case 4:
		case 5: return VLAN_2_NAME;
		case 6:
		default: return VLAN_3_NAME;
		}
	}
	else
	{
		switch (nIdx)
		{
		case 0:
		case 1: return QMIMUX_0_NAME;
		case 2:
		case 3: return QMIMUX_1_NAME;
		case 4:
		case 5: return QMIMUX_2_NAME;
		case 6:
		default: return QMIMUX_3_NAME;
		}
	}
}

typedef int(*packer)(pack_qmi_t*, uint8_t*, uint16_t*, void*);
typedef int(*unpacker)(uint8_t*, uint16_t, void*);

/*
 * Name:     SendReceive
 *
 * Purpose:  Send QMI request and receive QMI response.
 *
 * Params:   pService		- Pointer to QMI service object.
 *           fnPacker		- Pointer to packer function.
 *           pPackerName	- Packer function name.
 *           pInput			- Pointer to input argument structure.  If -1, no argument is required.
 *           fnUnpacker		- Pointer to packer function.
 *           pUnpackerName	- Unpacker function name.  If -1, not to show error if unpacking failed.
 *           pOutput		- Pointer to output structure.  If -1, no argument is required.
 *
 * Return:   SUCCESS	- If the transaction is successful
 *           FAILURE	- If the transaction fail
 *
 * Notes:    None
 */
static int SendReceive(QmiService* pService, packer fnPacker, char* pPackerName, void* pInput, unpacker fnUnpacker, char* pUnpackerName, void* pOutput)
{
	if (pService == NULL || fnPacker == NULL || fnUnpacker == NULL)
		return eLITE_CONNECT_APP_ERR_QMI;

	if (pInput == NULL)
	{
		dlog(eLOG_ERROR, "%s: invalid arg\n", pPackerName ? pPackerName : __func__);
		return eQCWWAN_ERR_INVALID_ARG;
	}

	if (pOutput == NULL)
	{
		dlog(eLOG_ERROR, "%s: invalid arg\n", pUnpackerName ? pUnpackerName : __func__);
		return eQCWWAN_ERR_INVALID_ARG;
	}

	QmiSyncObject syncObject;
	QmiSyncObject_Initialize(&syncObject);

	pack_qmi_t req_ctx;
	memset(&req_ctx, 0, sizeof(req_ctx));
	req_ctx.xid = QmiService_GetNextTransactionId(pService);

	int eLiteRet = eLITE_CONNECT_APP_ERR_QMI;

	if (fnPacker(&req_ctx, syncObject.buffer, &syncObject.bufferSize, pInput == (void*)-1 ? NULL : pInput) != 0)
		dlog(eLOG_ERROR, "%s error\n", pPackerName == NULL ? "pack" : pPackerName);
	else
	{
		QmiSyncObject_Lock(&syncObject);

		int ret = QmiService_SendRequest(
			pService,
			req_ctx.xid,
			syncObject.buffer,
			syncObject.bufferSize,
			QmiSyncObject_ResponseCallback,
			(void*)&syncObject
		);

		if (ret != SUCCESS)
		{
			QmiSyncObject_Unlock(&syncObject);
		}
		else
		{
			ret = QmiSyncObject_TimedWait(&syncObject, 10);
			QmiSyncObject_Unlock(&syncObject);

			if (ret != SUCCESS)
			{
				dlog(eLOG_WARN, "Function timed out\n");
				QmiService_CancelTransaction(pService, req_ctx.xid);
			}
			else
			{
				enum eQCWWANError qcError = fnUnpacker(syncObject.buffer, syncObject.bufferSize, pOutput == (void*)-1 ? NULL : pOutput);
				if (qcError == eQCWWAN_ERR_NONE)
					eLiteRet = eLITE_CONNECT_APP_OK;
				else if (pUnpackerName != (void*)-1)
					dlog(eLOG_INFO, "%s failed: %d\n", pUnpackerName == NULL ? "unpack" : pUnpackerName, qcError);
			}
		}
	}

	QmiSyncObject_Destroy(&syncObject);

	return eLiteRet;
}

/*
 * Name:     SendRequest
 *
 * Purpose:  Send QMI request.
 *
 * Params:   pService		- Pointer to QMI service object.
 *           fnPacker		- Pointer to packer function.
 *           pPackerName	- Packer function name.
 *           pInput			- Pointer to input argument structure.  If -1, no argument is required.
 *           fnCallback		- Pointer to callback function.
 *           pContext		- Context of callback.
 *           pPendingXid	- Pointer to receive transaction ID.
 *
 * Return:   SUCCESS	- If the transaction is successful
 *           FAILURE	- If the transaction fail
 *
 * Notes:    None
 */
int SendRequest(QmiService* pService, packer fnPacker, char* pPackerName, void* pInput, QMI_RESPONSE_CALLBACK fnCallback, void* pContext, uint16_t* pPendingXid)
{
	if (pService == NULL || fnPacker == NULL || fnCallback == NULL)
		return eQCWWAN_ERR_INVALID_ARG;

	if (pInput == NULL)
	{
		dlog(eLOG_ERROR, "%s: invalid arg\n", pPackerName ? pPackerName : __func__);
		return eQCWWAN_ERR_INVALID_ARG;
	}

	if (pPendingXid == NULL)
	{
		dlog(eLOG_ERROR, "%s: invalid arg\n", __func__);
		return eQCWWAN_ERR_INVALID_ARG;
	}

	pack_qmi_t req_ctx;
	memset(&req_ctx, 0, sizeof(req_ctx));
	req_ctx.xid = QmiService_GetNextTransactionId(pService);

	uint8_t buffer[MAX_QMI_PACKET_SIZE];    // Can be used for outgoing and incoming data
	uint16_t bufferSize; // Will be set to number of incoming bytes

	int eLiteRet = fnPacker(&req_ctx, buffer, &bufferSize, pInput == (void*)-1 ? NULL : pInput);

	if (eLiteRet != 0)
		dlog(eLOG_ERROR, "%s error %d\n", pPackerName == NULL ? "pack" : pPackerName, eLiteRet);
	else
	{
		if (*pPendingXid != 0)
		{
			QmiService_CancelTransaction(pService, *pPendingXid);
			*pPendingXid = 0;
		}

		eLiteRet = QmiService_SendRequest(pService, req_ctx.xid, buffer, bufferSize, fnCallback, pContext);

		if (eLiteRet == 0)
			*pPendingXid = req_ctx.xid;
	}

	return eLiteRet;
}

int unpackDmsGetModelID(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_dms_GetModelID_t *pModelId = (unpack_dms_GetModelID_t*)pOutput;
	int ret = unpack_dms_GetModelID(pResp, respLen, pModelId);
	if (ret == eQCWWAN_ERR_NONE)
		printf("ModelID: %s\n", pModelId->modelid);

	return ret;
}

static int get_model_id(unpack_dms_GetModelID_t *pModelId)
{
	return SendReceive(&s_DmsService, pack_dms_GetModelID, "pack_dms_GetModelID", (void*)-1, unpackDmsGetModelID, "unpack_dms_GetModelID", pModelId);
}

int packGetHomeNetwork(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pInput == NULL ? pack_nas_GetHomeNetwork(pCtx, pReqBuf, pLen) : eQCWWAN_ERR_INVALID_ARG;
}

int unpackGetHomeNetwork(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_nas_GetHomeNetwork_t *pHomeNw = (unpack_nas_GetHomeNetwork_t*)pOutput;
	int ret = unpack_nas_GetHomeNetwork(pResp, respLen, pHomeNw);
	if (ret == eQCWWAN_ERR_NONE)
		printf("HomeNetwork: %s\n", pHomeNw->name);

	return ret;
}

static int get_home_network(unpack_nas_GetHomeNetwork_t *pHomeNw)
{
	return SendReceive(&s_NasService, packGetHomeNetwork, "pack_nas_GetHomeNetwork", (void*)-1, unpackGetHomeNetwork, "unpack_nas_GetHomeNetwork", pHomeNw);
}

// ------------------------------------------------
int packNasGetSysSelectionPrefExt(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	(void)pInput;
	return pack_nas_SLQSGetSysSelectionPrefExt(pCtx, pReqBuf, pLen);
}

int unpackNasGetSysSelectionPrefExt(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_nas_SLQSGetSysSelectionPrefExt_t* pNetSelPref = (unpack_nas_SLQSGetSysSelectionPrefExt_t*)pOutput;
	int ret = unpack_nas_SLQSGetSysSelectionPrefExt(pResp, respLen, pNetSelPref);
	if (ret == eQCWWAN_ERR_NONE)
	{
		printf("Network Selection Preference: ");
		if (swi_uint256_get_bit(pNetSelPref->ParamPresenceMask, 0x16))
			printf("%s\n", *pNetSelPref->pNetSelPref == 0 ? "auto" : "manual");
		else
			printf("unknown\n");
	}

	return ret;
}

int get_network_selection_pref(unpack_nas_SLQSGetSysSelectionPrefExt_t* pOutput)
{
	return SendReceive(&s_NasService,
		packNasGetSysSelectionPrefExt, "pack_nas_SLQSGetSysSelectionPrefExt", (void*)-1,
		unpackNasGetSysSelectionPrefExt, "unpack_nas_SLQSGetSysSelectionPrefExt", pOutput);
}

// ------------------------------------------------
int packNasSetSysSelectionPrefExt(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_nas_SLQSSetSysSelectionPrefExt(pCtx, pReqBuf, pLen, (pack_nas_SLQSSetSysSelectionPrefExt_t*)pInput);
}

int unpackNasSetSysSelectionPrefExt(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_nas_SLQSSetSysSelectionPrefExt(pResp, respLen, (unpack_nas_SLQSSetSysSelectionPrefExt_t*)pOutput);
}

int set_network_selection_pref(pack_nas_SLQSSetSysSelectionPrefExt_t* reqArg, unpack_nas_SLQSSetSysSelectionPrefExt_t* pOutput)
{
	return SendReceive(&s_NasService, 
		packNasSetSysSelectionPrefExt, "pack_nas_SLQSSetSysSelectionPrefExt", reqArg,
		unpackNasSetSysSelectionPrefExt, "unpack_nas_SLQSSetSysSelectionPrefExt", pOutput);
}

int packGetMitDevList(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	UNUSEDPARAM(pInput);
	return pack_tmd_SLQSTmdGetMitigationDevList(pCtx, pReqBuf, pLen);
}

int unpackGetMitDevList(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_tmd_SLQSTmdGetMitigationDevList(pResp, respLen, (unpack_tmd_SLQSTmdGetMitigationDevList_t*)pOutput);
}

static int get_mit_devlist(unpack_tmd_SLQSTmdGetMitigationDevList_t *pDevList)
{
	return SendReceive(&s_TmdService, packGetMitDevList, "pack_tmd_SLQSTmdGetMitigationDevList", (void*)-1,
		unpackGetMitDevList, "unpack_tmd_SLQSTmdGetMitigationDevList", pDevList);
}

int packRegNotifMitLvl(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_tmd_SLQSTmdRegNotMitigationLvl(pCtx, pReqBuf, pLen, pInput);
}

int unpackRegNotifMitLvl(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	UNUSEDPARAM(pOutput);
	unpack_tmd_SLQSTmdRegNotMitigationLvl_t output;
	return unpack_tmd_SLQSTmdRegNotMitigationLvl(pResp, respLen, &output);
}

static int reg_notif_mit_lvl(pack_tmd_SLQSTmdRegNotMitigationLvl_t *pDev)
{
	return SendReceive(&s_TmdService, packRegNotifMitLvl, "pack_tmd_SLQSTmdRegNotMitigationLvl", (void*)pDev,
		unpackRegNotifMitLvl, "unpack_tmd_SLQSTmdRegNotMitigationLvl", (void*)-1);
}

int packDeRegNotifMitLvl(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_tmd_SLQSTmdDeRegNotMitigationLvl(pCtx, pReqBuf, pLen, pInput);
}

int unpackDeRegNotifMitLvl(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	UNUSEDPARAM(pOutput);
	unpack_tmd_SLQSTmdDeRegNotMitigationLvl_t output;
	return unpack_tmd_SLQSTmdDeRegNotMitigationLvl(pResp, respLen, &output);
}

static int dereg_notif_mit_lvl(pack_tmd_SLQSTmdDeRegNotMitigationLvl_t *pDev)
{
	return SendReceive(&s_TmdService, packDeRegNotifMitLvl, "pack_tmd_SLQSTmdDeRegNotMitigationLvl", (void*)pDev,
		unpackDeRegNotifMitLvl, "unpack_tmd_SLQSTmdDeRegNotMitigationLvl", (void*)-1);
}

int packGetTsSensorList(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	UNUSEDPARAM(pInput);
	return pack_ts_SLQSTsGetSensorList(pCtx, pReqBuf, pLen);
}

int unpackGetTsSensorList(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_ts_SLQSTsGetSensorList(pResp, respLen, (unpack_ts_SLQSTsGetSensorList_t*)pOutput);
}

static int get_ts_sensorlist(unpack_ts_SLQSTsGetSensorList_t *pSensorList)
{
	return SendReceive(&s_TsService, packGetTsSensorList, "pack_ts_SLQSTsGetSensorList", (void*)-1,
		unpackGetTsSensorList, "unpack_ts_SLQSTsGetSensorList", pSensorList);
}

int packRegNotifSensorTemp(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_ts_SLQSTsRegNotTempRpt(pCtx, pReqBuf, pLen, pInput);
}

int unpackRegNotifSensorTemp(uint8_t* pResp, uint16_t respLen, void* pInput)
{
	UNUSEDPARAM(pInput);
	return unpack_ts_SLQSTsRegNotTempRpt(pResp, respLen);
}

static int reg_notif_sensor_temp(pack_ts_SLQSTsRegNotTempRpt_t *pSensor)
{
	return SendReceive(&s_TsService, packRegNotifSensorTemp, "pack_ts_SLQSTsRegNotTempRpt", (void*)pSensor,
		unpackRegNotifSensorTemp, "unpack_ts_SLQSTsRegNotTempRpt", (void*)-1);
}

static void register_mit_ind(bool reg)
{
	dlog(eLOG_INFO, "\n%s thermal mitigation device notification(s)\n", reg ? "Registering" : "Deregistering");

	if (reg)
	{
		memset(&tmdDevList, 0, sizeof(unpack_tmd_SLQSTmdGetMitigationDevList_t));
		tmdDevList.MitigationDevListLen = TMD_MAX_DEV_LIST;

		int rtn = get_mit_devlist(&tmdDevList);
		if (rtn != eLITE_CONNECT_APP_OK)
		{
			tmdDevList.MitigationDevListLen = 0;
			dlog(eLOG_ERROR, "Unable to get mitigation device list - %d\n", rtn);
			return;
		}
	}

	for (uint8_t i = 0; i < tmdDevList.MitigationDevListLen; i++)
	{
		int rtn = reg ?
			reg_notif_mit_lvl((pack_tmd_SLQSTmdRegNotMitigationLvl_t*)&tmdDevList.MitigationDevList[i]) :
			dereg_notif_mit_lvl((pack_tmd_SLQSTmdDeRegNotMitigationLvl_t*)&tmdDevList.MitigationDevList[i]);
		
		if (rtn == 0)
			dlog(eLOG_INFO, "  Thermal mitigation device \"%s\" notification %s\n",
				tmdDevList.MitigationDevList[i].mitigationDevId, reg ? "registered" : "deregistered");
		else
			dlog(eLOG_ERROR, "  Thermal mitigation device \"%s\" notification %s error %d\n",
				tmdDevList.MitigationDevList[i].mitigationDevId, reg ? "registeration" : "deregisteration", rtn);
	}
}

static void register_ts_ind(bool reg)
{
	dlog(eLOG_INFO, "\n%s thermal sensor notification(s)\n", reg ? "Registering" : "Deregistering");

	if (reg)
	{
		memset(&tsSensorList, 0, sizeof(unpack_ts_SLQSTsGetSensorList_t));
		tsSensorList.sensorListLen = MAX_SENSOR_LIST_LEN;

		int rtn = get_ts_sensorlist(&tsSensorList);
		if (rtn != eLITE_CONNECT_APP_OK)
		{
			tsSensorList.sensorListLen = 0;
			dlog(eLOG_ERROR, "Unable to get thermal sensor list - %d\n", rtn);
			return;
		}
	}

	for (uint8_t i = 0; i < tsSensorList.sensorListLen; i++)
	{
		float highThres = 50.0;
		float lowThres = 10.0;
		pack_ts_SLQSTsRegNotTempRpt_t sRegNotif;
		memcpy(&sRegNotif.sensor, &tsSensorList.sensorList[i], sizeof(ts_sensorId));
		sRegNotif.curTempReport = 1;
		sRegNotif.pHighThresTemp = reg ? &highThres : NULL;
		sRegNotif.pLowThresTemp = reg ? &lowThres : NULL;
		sRegNotif.pSeqNum = NULL;

		int rtn = reg_notif_sensor_temp(&sRegNotif);

		if (rtn == 0)
			dlog(eLOG_INFO, "  Thermal sensor \"%s\" notification %s\n",
				sRegNotif.sensor.sensorId, reg ? "registered" : "deregistered");
		else
			dlog(eLOG_ERROR, "  Thermal sensor \"%s\" notification %s error %d\n",
				sRegNotif.sensor.sensorId, reg ? "registeration" : "deregisteration", rtn);
	}
}

int packGetSessionState(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pInput == NULL ? pack_wds_GetSessionState(pCtx, pReqBuf, pLen) : eQCWWAN_ERR_INVALID_ARG;
}

int unpackGetSessionState(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_wds_GetSessionState_t *pSessionState = (unpack_wds_GetSessionState_t*)pOutput;
	return unpack_wds_GetSessionState(pResp, respLen, pSessionState);
}

static int get_session_state(unpack_wds_GetSessionState_t *pSessionState, int nIdx)
{
	return SendReceive(&sessions[nIdx].wdsSvc, packGetSessionState, "pack_wds_GetSessionState", (void*)-1, unpackGetSessionState, "unpack_wds_GetSessionState", pSessionState);
}

int packGetRuntimeSettings(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSGetRuntimeSettings(pCtx, pReqBuf, pLen, (pack_wds_SLQSGetRuntimeSettings_t*)pInput);
}

int unpackGetRuntimeSettings(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSGetRuntimeSettings(pResp, respLen, (unpack_wds_SLQSGetRuntimeSettings_t*)pOutput);
}

static int get_runtime_settings(
	pack_wds_SLQSGetRuntimeSettings_t* pRuntimeSettings,
	unpack_wds_SLQSGetRuntimeSettings_t* pRuntimeSettingsOut,
	int nIdx)
{
	return SendReceive(&sessions[nIdx].wdsSvc,
		packGetRuntimeSettings, "pack_wds_SLQSGetRuntimeSettings", pRuntimeSettings,
		unpackGetRuntimeSettings, "unpack_wds_SLQSGetRuntimeSettings", pRuntimeSettingsOut);
}

int packGetPktStats(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_GetPacketStatistics(pCtx, pReqBuf, pLen, (pack_wds_GetPacketStatistics_t*)pInput);
}

int unpackGetPktStats(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_GetPacketStatistics(pResp, respLen, (unpack_wds_GetPacketStatistics_t*)pOutput);
}

static int getPktStats(
	pack_wds_GetPacketStatistics_t *pPktStatsIn,
	unpack_wds_GetPacketStatistics_t *pPktStatsOut)
{
	return SendReceive(&sessions[0].wdsSvc,
		packGetPktStats, "pack_wds_GetPacketStatistics", pPktStatsIn,
		unpackGetPktStats, "unpack_wds_GetPacketStatistics", pPktStatsOut);
}

/******************************************************************************
* Option 8 : Delete a profile stored on the device
******************************************************************************/

int packDeleteProfile(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSDeleteProfile(pCtx, pReqBuf, pLen, (pack_wds_SLQSDeleteProfile_t*)pInput);
}

int unpackDeleteProfile(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSDeleteProfile(pResp, respLen, (unpack_wds_SLQSDeleteProfile_t*)pOutput);
}

static int delete_profile(
	pack_wds_SLQSDeleteProfile_t *pProfileIn,
	unpack_wds_SLQSDeleteProfile_t *pProfileOut)
{
	return SendReceive(&sessions[0].wdsSvc,
		packDeleteProfile, "pack_wds_SLQSDeleteProfile", pProfileIn,
		unpackDeleteProfile, "unpack_wds_SLQSDeleteProfile", pProfileOut);
}

/*
 * Name:     delete_profile_from_device
 *
 * Purpose:  Delete the profile from the device for the profile id provided by
 *           the user.
 *
 * Params:   None
 *
 * Return:   None.
 *
 * Notes:    None
 */
void delete_profile_from_device()
{
	while (1)
	{
		/* Display all the profiles stored on the device */
		display_all_profiles(true);

		/* If no profile exist on the device, return */
		if (0 == indexInfo.totalProfilesOnDevice)
		{
			dlog(eLOG_ERROR, "No Profile exist on the device for deletion or check device connectivity\n\n");
			break;
		}

		/* Receive the user input */
		uint8_t profileId = GetUserProfileId();

		/* If only <ENTER> is pressed by the user, return to main menu */
		if (ENTER_KEY_PRESSED == profileId)
			break;

		pack_wds_SLQSDeleteProfile_t   deleteProfile;
		memset(&deleteProfile, 0, sizeof(deleteProfile));
		deleteProfile.profileIndex = profileId;
		deleteProfile.profileType = PROFILE_TYPE_UMTS;

		unpack_wds_SLQSDeleteProfile_t   deleteProfileOut;
		memset(&deleteProfileOut, 0, sizeof(deleteProfileOut));
		deleteProfileOut.extendedErrorCode = (uint16_t)-1;

		/* Delete the profile from the device */
		int rtn = delete_profile(&deleteProfile, &deleteProfileOut);

		if (rtn != eLITE_CONNECT_APP_OK)
			dlog(eLOG_ERROR, "Profile deletion Failed\nFailure cause - %d\n", rtn);
		else
			dlog(eLOG_INFO, "Profile for index %d deleted successfully\n", profileId);
	}
}

int packNasPerformNetworkScan(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	(void)pInput;
	return pack_nas_PerformNetworkScan(pCtx, pReqBuf, pLen);
}

const char* GetInuse(uint32_t val)
{
	switch (val)
	{
	case 1:
		return "Current serving";
	case 2:
		return "Available";
	default:
		break;
	}

	return "Unknown";
}

const char* GetForbidden(uint32_t val)
{
	switch (val)
	{
	case 1:
		return "Forbidden";
	case 2:
		return "Not forbidden";
	default:
		break;
	}

	return "Unknown";
}

const char* GetPreferred(uint32_t val)
{
	switch (val)
	{
	case 1:
		return "Preferred";
	case 2:
		return "Not preferred";
	default:
		break;
	}

	return "Unknown";
}

const char* GetRoaming(uint32_t val)
{
	switch (val)
	{
	case 1:
		return "Home";
	case 2:
		return "Roam";
	default:
		break;
	}

	return "Unknown";
}

void DisplayPerformNetworkScanResult(unpack_nas_PerformNetworkScan_t *pNetworkScan)
{
	printf("\nNetwork Scan Result:");

	if (swi_uint256_get_bit(pNetworkScan->ParamPresenceMask, 0x13) && pNetworkScan->pScanResult)
		printf(" %d\n", *pNetworkScan->pScanResult);
	else
		printf("\n");

	static const char* ratGERAN = "GERAN";	// 0x04
	static const char* ratUMTS = "UMTS";	// 0x05
	static const char* ratLTE = "LTE";	// 0x08
	static const char* ratTDSCDMA = "TD-SCDMA";	// 0x09
	static const char* ratUnknown = "Unknown";

	for (uint8_t i = 0;
		swi_uint256_get_bit(pNetworkScan->ParamPresenceMask, 0x10) &&
		pNetworkScan->p3GppNetworkInfoInstances &&
		pNetworkScan->p3GppNetworkInstanceSize &&
		i < *pNetworkScan->p3GppNetworkInstanceSize;
		i++)
	{
		printf("   Network %d: %s\n", i, pNetworkScan->p3GppNetworkInfoInstances[i].Desription);
		printf("      MCC/MNC: %d/%d\n", pNetworkScan->p3GppNetworkInfoInstances[i].MCC, pNetworkScan->p3GppNetworkInfoInstances[i].MNC);
		printf("      In Use: %s(%d)\n", GetInuse(pNetworkScan->p3GppNetworkInfoInstances[i].InUse), pNetworkScan->p3GppNetworkInfoInstances[i].InUse);
		printf("      Forbidden: %s(%d)\n", GetForbidden(pNetworkScan->p3GppNetworkInfoInstances[i].Forbidden), pNetworkScan->p3GppNetworkInfoInstances[i].Forbidden);
		printf("      Preferred: %s(%d)\n", GetPreferred(pNetworkScan->p3GppNetworkInfoInstances[i].Preferred), pNetworkScan->p3GppNetworkInfoInstances[i].Preferred);
		printf("      Roaming: %s(%d)\n", GetRoaming(pNetworkScan->p3GppNetworkInfoInstances[i].Roaming), pNetworkScan->p3GppNetworkInfoInstances[i].Roaming);

		if (swi_uint256_get_bit(pNetworkScan->ParamPresenceMask, 0x11) &&
			pNetworkScan->pRATINstance &&
			pNetworkScan->pRATInstanceSize &&
			i < *pNetworkScan->pRATInstanceSize)
		{
			const char* pRat = ratUnknown;
			switch (pNetworkScan->pRATINstance[i].RAT)
			{
			case 4:
				pRat = ratGERAN;
				break;
			case 5:
				pRat = ratUMTS;
				break;
			case 8:
				pRat = ratLTE;
				break;
			case 9:
				pRat = ratTDSCDMA;
				break;
			default:
				break;
			}

			printf("      RAT: %s(%d)", pRat, pNetworkScan->pRATINstance[i].RAT);

			if (pNetworkScan->p3GppNetworkInfoInstances[i].MCC == pNetworkScan->pRATINstance[i].MCC &&
				pNetworkScan->p3GppNetworkInfoInstances[i].MNC == pNetworkScan->pRATINstance[i].MNC)
				printf("\n");
			else
				printf(" - MCC/MNC(%d/%d) doesn't match\n", pNetworkScan->pRATINstance[i].MCC, pNetworkScan->pRATINstance[i].MNC);
		}

		if (swi_uint256_get_bit(pNetworkScan->ParamPresenceMask, 0x12) &&
			pNetworkScan->pPCSInstance &&
			pNetworkScan->pPCSInstanceSize &&
			i < *pNetworkScan->pPCSInstanceSize)
		{
			printf("      MNC includes PCS digit: %s", pNetworkScan->pPCSInstance[i].includes_pcs_digit ? "TRUE" : "FALSE");

			if (pNetworkScan->p3GppNetworkInfoInstances[i].MCC == pNetworkScan->pPCSInstance[i].MCC &&
				pNetworkScan->p3GppNetworkInfoInstances[i].MNC == pNetworkScan->pPCSInstance[i].MNC)
				printf("\n");
			else
				printf(" - MCC/MNC(%d/%d) doesn't match\n", pNetworkScan->pPCSInstance[i].MCC, pNetworkScan->pPCSInstance[i].MNC);
		}
	}

	if (swi_uint256_get_bit(pNetworkScan->ParamPresenceMask, 0x17) && pNetworkScan->pPCIInfo)
	{
		for (uint8_t i = 0; i < pNetworkScan->pPCIInfo->PCICellInfoLen; i++)
		{
			printf("   PCI Cell Info %d: freq: %d, Cell ID: %d, Global Cell ID: %d\n", i,
				pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].freq,
				pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].cellID,
				pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].GlobalCellID);

			for (uint8_t j = 0; j < pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].PlmnLen; j++)
			{
				printf("    PLMN %d: MCC:%d, MNC:%d, Includes PCS digit: %d\n", j,
					pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].nasQmisNasPcsDigit[j].MCC,
					pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].nasQmisNasPcsDigit[j].MNC,
					pNetworkScan->pPCIInfo->nasQmisNasSlqsNasPCICellInfo[i].nasQmisNasPcsDigit[j].includes_pcs_digit);
			}
		}

		printf("   PCI Cell RSRP: %d, RSRP Rx0: %d, RSRP Rx1: %d, RSRQ: %d, RSRQ Rx0: %d, RSRQ Rx1: %d\n",
			pNetworkScan->pPCIInfo->rsrp, pNetworkScan->pPCIInfo->rsrpRx0, pNetworkScan->pPCIInfo->rsrpRx1,
			pNetworkScan->pPCIInfo->rsrq, pNetworkScan->pPCIInfo->rsrqRx0, pNetworkScan->pPCIInfo->rsrqRx1);
	}

	for (uint8_t i = 0; swi_uint256_get_bit(pNetworkScan->ParamPresenceMask, 0x18) && pNetworkScan->pLteOpModeTlv && i < pNetworkScan->pLteOpModeTlv->lteOpModeLen; i++)
	{
		printf("   LTE Op Mode %d: MCC/MNC: %d/%d, Mode: %d\n", i,
			pNetworkScan->pLteOpModeTlv->MCC[i], pNetworkScan->pLteOpModeTlv->MNC[i], pNetworkScan->pLteOpModeTlv->lteOpMode[i]);
	}

	printf("\n");
}

void ScanNetworkCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pContext)
{
	(void)pContext;

	g_NasPendingXid = 0;

	uint8_t	                   NetworkInstanceSize = 1;
	uint8_t                    RATInstanceSize = 1;
	uint8_t                    PCSInstanceSize = 1;
	uint32_t                   ScanResult = 0;
	nas_QmisNasSlqsNasPCIInfo  PCIInfo;
	nas_lteOpModeTlv           LteOpModeTlv;

	unpack_nas_PerformNetworkScan_t output;
	memset(&output, 0, sizeof(output));
	output.p3GppNetworkInstanceSize = &NetworkInstanceSize;
	output.pRATInstanceSize = &RATInstanceSize;
	output.pPCSInstanceSize = &PCSInstanceSize;
	output.pScanResult = &ScanResult;
	output.pPCIInfo = &PCIInfo;
	output.pLteOpModeTlv = &LteOpModeTlv;

	int ret = 0;

	do
	{
		if (output.p3GppNetworkInfoInstances)
			free(output.p3GppNetworkInfoInstances);

		output.p3GppNetworkInfoInstances = NetworkInstanceSize == 0 ? NULL : malloc(sizeof(nas_QmiNas3GppNetworkInfo)*NetworkInstanceSize);

		if (NetworkInstanceSize > 0 && output.p3GppNetworkInfoInstances == NULL)
		{
			ret = eQCWWAN_ERR_MEMORY;
			break;
		}

		if (output.pRATINstance)
			free(output.pRATINstance);

		output.pRATINstance = RATInstanceSize == 0 ? NULL : malloc(sizeof(nas_QmiNas3GppNetworkRAT)*RATInstanceSize);

		if (RATInstanceSize > 0 && output.pRATINstance == NULL)
		{
			ret = eQCWWAN_ERR_MEMORY;
			break;
		}

		if (output.pPCSInstance)
			free(output.pPCSInstance);

		output.pPCSInstance = PCSInstanceSize == 0 ? NULL : malloc(sizeof(nas_QmisNasPcsDigit)*PCSInstanceSize);

		if (PCSInstanceSize > 0 && output.pPCSInstance == NULL)
		{
			ret = eQCWWAN_ERR_MEMORY;
			break;
		}

		ret = unpack_nas_PerformNetworkScan(qmiPacket, qmiPacketSize, &output);
		if (ret != eQCWWAN_ERR_NONE && ret != eQCWWAN_ERR_BUFFER_SZ)
			printf("unpack_nas_PerformNetworkScan: unpacking function error %d\n", ret);
	} while (ret == eQCWWAN_ERR_BUFFER_SZ);

	if (ret == eQCWWAN_ERR_NONE)
		DisplayPerformNetworkScanResult(&output);
	else
		printf("unpack_nas_PerformNetworkScan errorCode: %d - %s\n", ret, helper_get_error_reason(ret));

	if (output.p3GppNetworkInfoInstances)
		free(output.p3GppNetworkInfoInstances);

	if (output.pRATINstance)
		free(output.pRATINstance);

	if (output.pPCSInstance)
		free(output.pPCSInstance);
}

void scan_available_networks()
{
	int ret = SendRequest(&s_NasService, packNasPerformNetworkScan, "pack_nas_PerformNetworkScan", (void*)-1, ScanNetworkCallback, NULL, &g_NasPendingXid);
	if (ret != SUCCESS)
		printf("QmiService_SendRequest returned error %d\n", ret);
	else
		printf("NetworkScan may take a few minutes.  Please wait...\n");
}

void get_pkt_stats()
{
	uint32_t statmask = 0xFF;
	pack_wds_GetPacketStatistics_t sGetPktStatisticsReq;
	memset(&sGetPktStatisticsReq, 0, sizeof(sGetPktStatisticsReq));
	sGetPktStatisticsReq.pStatMask = &statmask;

	uint32_t TXPacketSuccesses;
	uint32_t RXPacketSuccesses;
	uint32_t TXPacketErrors;
	uint32_t RXPacketErrors;
	uint32_t TXPacketOverflows;
	uint32_t RXPacketOverflows;
	uint64_t TXOkBytesCount;
	uint64_t RXOkBytesCount;
	uint64_t TXOKBytesLastCall;
	uint64_t RXOKBytesLastCall;
	uint32_t TXDroppedCount;
	uint32_t RXDroppedCount;
	unpack_wds_GetPacketStatistics_t sGetPktStatisticsResp;
	memset(&sGetPktStatisticsResp, 0, sizeof(sGetPktStatisticsResp));
	sGetPktStatisticsResp.pTXPacketSuccesses = &TXPacketSuccesses;
	sGetPktStatisticsResp.pRXPacketSuccesses = &RXPacketSuccesses;
	sGetPktStatisticsResp.pTXPacketErrors = &TXPacketErrors;
	sGetPktStatisticsResp.pRXPacketErrors = &RXPacketErrors;
	sGetPktStatisticsResp.pTXPacketOverflows = &TXPacketOverflows;
	sGetPktStatisticsResp.pRXPacketOverflows = &RXPacketOverflows;
	sGetPktStatisticsResp.pTXOkBytesCount = &TXOkBytesCount;
	sGetPktStatisticsResp.pRXOkBytesCount = &RXOkBytesCount;
	sGetPktStatisticsResp.pTXOKBytesLastCall = &TXOKBytesLastCall;
	sGetPktStatisticsResp.pRXOKBytesLastCall = &RXOKBytesLastCall;
	sGetPktStatisticsResp.pTXDroppedCount = &TXDroppedCount;
	sGetPktStatisticsResp.pRXDroppedCount = &RXDroppedCount;

	int rtn = getPktStats(&sGetPktStatisticsReq, &sGetPktStatisticsResp);

	if (rtn != eLITE_CONNECT_APP_OK)
		dlog(eLOG_ERROR, "Get packet statistics Failed\nFailure cause - %d\n", rtn);
	else
	{
		printf("\nPacket Statistics:\n");

		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x10))
			printf("\tTx Packets OK: %d\n", TXPacketSuccesses);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x11))
			printf("\tRx Packets OK: %d\n", RXPacketSuccesses);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x12))
			printf("\tTx Packets Errors: %d\n", TXPacketErrors);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x13))
			printf("\tRx Packets Errors: %d\n", RXPacketErrors);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x14))
			printf("\tTx Overflows: %d\n", TXPacketOverflows);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x15))
			printf("\tRx Overflows: %d\n", RXPacketOverflows);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x19))
			printf("\tTx Bytes OK: %"PRIu64"\n", TXOkBytesCount);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x1A))
			printf("\tRx Bytess OK: %"PRIu64"\n", RXOkBytesCount);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x1B))
			printf("\tLast Call Tx Bytes OK: %"PRIu64"\n", TXOKBytesLastCall);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x1C))
			printf("\tLast Call Rx Bytess OK: %"PRIu64"\n", RXOKBytesLastCall);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x1D))
			printf("\tTx Packets Dropped: %d\n", TXDroppedCount);
		if (swi_uint256_get_bit(sGetPktStatisticsResp.ParamPresenceMask, 0x1E))
			printf("\tRx Packets Dropped: %d\n", RXDroppedCount);
	}
}

int PackGetCurrentChannelRate(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	UNUSEDPARAM(pInput);
	int ret = pack_wds_SLQSSwiGetCurrentChannelRate(pCtx, pReqBuf, pLen);
	if (ret != eQCWWAN_ERR_NONE)
		dlog(eLOG_ERROR, "pack_wds_SLQSSwiGetCurrentChannelRate rtn = %d\n", ret);
	return ret;
}

int UnpackGetCurrentChannelRate(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_wds_SLQSGetCurrentChannelRate_t *pGetCurrentChannelRate = (unpack_wds_SLQSGetCurrentChannelRate_t*)pOutput;
	int ret = unpack_wds_SLQSSwiGetCurrentChannelRate(pResp, respLen, pGetCurrentChannelRate);
	if (ret != eQCWWAN_ERR_NONE)
		dlog(eLOG_ERROR, "unpack_wds_SLQSSwiGetCurrentChannelRate rtn = %d\n", ret);
	return ret;
}

/*
 * Name:     get_Current_channel_rate
 *
 * Purpose:  Get Current Channel Rate.
 *
 * Params:   None
 *
 * Return:   None.
 *
 * Notes:    None
 */
void get_Current_channel_rate()
{
	unpack_wds_SLQSGetCurrentChannelRate_t sCurrentChannelRate;
	memset(&sCurrentChannelRate, 0, sizeof(unpack_wds_SLQSGetCurrentChannelRate_t));

	int ret = SendReceive(&sessions[0].wdsSvc, PackGetCurrentChannelRate, "pack_wds_SLQSSwiGetCurrentChannelRate", (void*)-1, UnpackGetCurrentChannelRate, "unpack_wds_SLQSSwiGetCurrentChannelRate", &sCurrentChannelRate);

	if (ret == eQCWWAN_ERR_NONE)
	{
		printf("\nGet Current Channel Rate\n");
		printf("\tcurrent channel tx rate:  %d\n", sCurrentChannelRate.current_channel_tx_rate);
		printf("\tcurrent channel rx rate:  %d\n", sCurrentChannelRate.current_channel_rx_rate);
		printf("\tmax channel tx rate:  %d\n", sCurrentChannelRate.max_channel_tx_rate);
		printf("\tmax channel rx rate:  %d\n", sCurrentChannelRate.max_channel_rx_rate);
	}
	else
	{
		dlog(eLOG_DEBUG, "Failed to get current channel rate\n");
		printf("\nFailed to Get Current Channel Rate\n");
	}
}

int PackQosSetEventCallback(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	int ret = pack_qos_SLQSSetQosEventCallback(pCtx, pReqBuf, pLen, (pack_qos_SLQSSetQosEventCallback_t*)pInput);
	dlog(eLOG_DEBUG, "pack_qos_SLQSSetQosEventCallback - rtn: %d\n", ret);

	return ret;
}

int UnpackQosSetEventCallback(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_qos_SLQSSetQosEventCallback_t *pSetQosEventCallback = (unpack_qos_SLQSSetQosEventCallback_t*)pOutput;
	int ret = unpack_qos_SLQSSetQosEventCallback(pResp, respLen, pOutput);
	if (ret == eQCWWAN_ERR_NONE && pSetQosEventCallback)
		dlog(eLOG_DEBUG, "unpack_qos_SLQSSetQosEventCallback - trn: %d\n", ret);

	return ret;
}

/*
 * Name:     qos_enable_qos_event
 *
 * Purpose:  enable/disable QoS events.
 *
 * Params:   None
 *
 * Return:   None.
 *
 * Notes:    None
 */
void qos_enable_qos_event(bool bEnable)
{
	pack_qos_SLQSSetQosEventCallback_t  sPackSetQosEventCallback;
	sPackSetQosEventCallback.network_supported_qos_profile_change_reporting = bEnable ? 1 : 0;
	sPackSetQosEventCallback.ext_technology_preference = 0x8001;

	unpack_qos_SLQSSetQosEventCallback_t sUnpackSetQosEventCallback;
	sUnpackSetQosEventCallback.Tlvresult = 0;

	int ret = SendReceive(&s_QosService, PackQosSetEventCallback, "pack_qos_SLQSSetQosEventCallback", &sPackSetQosEventCallback, UnpackQosSetEventCallback, "unpack_qos_SLQSSetQosEventCallback", &sUnpackSetQosEventCallback);
	if (ret != eQCWWAN_ERR_NONE)
		dlog(eLOG_ERROR, "SetQosEventCallback rtn = %d\n", ret);
}

int PackQosRequestQosEx(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	int ret = pack_qos_SLQSQosExRequest(pCtx, pReqBuf, pLen, (pack_qos_SLQSRequestQosExReq_t*)pInput);
	if (ret != eQCWWAN_ERR_NONE)
		dlog(eLOG_ERROR, "pack_qos_SLQSQosExRequest: %d - %s\n", ret, helper_get_error_reason(ret));

	return ret;
}

int UnpackQosRequestQosEx(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	int ret = unpack_qos_SLQSQosExRequest(pResp, respLen, (unpack_qos_SLQSQosRequestQosExResp_t*)pOutput);
	if (ret != eQCWWAN_ERR_NONE)
		dlog(eLOG_ERROR, "unpack_qos_SLQSQosExRequest: %d - %s\n", ret, helper_get_error_reason(ret));

	return ret;
}

static void GetQosFlow(pack_qos_QosFlow_t *pQosFlow)
{
	pQosFlow->flow_status = GetNumericValue(NULL, "Current flow status (Activated status=1 (default), Suspended status=2): ", g_enterActionDefault, 1, 1, 2);
    pQosFlow->flow_valid_params = GetNumericValue64(NULL, "Valid parameters mask: ", g_enterActionZero, 0, 0, MAX_UINT64_VALUE_SIZE);
    pQosFlow->ip_flow_trf_cls = GetNumericValue(NULL, "IP traffic class (conversational=0 streaming=1 interactive=2 background=3): ", g_enterActionZero, 0, 0, 3);
    pQosFlow->data_rate_max = GetNumericValue64(NULL, "Maximum required data rate (bits per second): ", g_enterActionZero, 0, 0, MAX_UINT64_VALUE_SIZE);
    pQosFlow->guaranteed_rate = GetNumericValue64(NULL, "Minimum guaranteed data rate (bits per second): ", g_enterActionZero, 0, 0, MAX_UINT64_VALUE_SIZE);
	pQosFlow->peak_rate = GetNumericValue(NULL, "Maximum rate at which data can be transmitted when the token bucket is full (bits per second): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->token_rate = GetNumericValue(NULL, "Rate at which tokens are put in the token bucket (bits per second): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->bucket_size = GetNumericValue(NULL, "Maximum number of tokens that can be accumulated at any instance (bytes): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_latency = GetNumericValue(NULL, "Maximum delay (in milliseconds): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_jitter = GetNumericValue(NULL, "Difference between the maximum and minimum latency (in milliseconds): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_pkt_error_rate_multiplier = GetNumericValue(NULL, "Factor m in calculating packet error rate: E = m*10**(-p): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_pkt_error_rate_exponen = GetNumericValue(NULL, "Factor p in calculating packet error rate: E = m*10**(-p): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_min_policed_packet_size = GetNumericValue(NULL, "Integer that defines the minimum packet size (in bytes) that will be policed for QoS guarantees: ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
	pQosFlow->ip_flow_max_allowed_packet_size = GetNumericValue(NULL, "Integer that defines the maximum packet size (in bytes) allowed in the IP flow: ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_3gpp_residual_bit_error_rate = GetNumericValue(NULL, "The undetected BER for each IP flow in the delivered packets (0 to 8): ", g_enterActionZero, 0, 0, 8);
    pQosFlow->ip_flow_3gpp_traffic_handling_priority = GetNumericValue(NULL, "Relative priority of the flow (0 to 3): ", g_enterActionZero, 0, 0, 3);
    pQosFlow->ip_flow_3gpp2_profile_id = GetNumericValue(NULL, "Profile ID shorthand for a defined set of QoS flow parameters (CDMA): ", g_enterActionZero, 0, 0, 16);
    pQosFlow->ip_flow_3gpp2_flow_priority = GetNumericValue(NULL, "Flow priority used by the network (CDMA): ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
    pQosFlow->ip_flow_3gpp_im_cn_flag = GetNumericValue(NULL, "IM CN subsystem signaling flag (0: FALSE (default), 1: TRUE): ", g_enterActionDefault, 0, 0, 1);
    pQosFlow->ip_flow_3gpp_sig_ind = GetNumericValue(NULL, "This parameter applies only to 3GPP networks (0: FALSE (default), 1: TRUE): ", g_enterActionDefault, 0, 0, 1);
    pQosFlow->ip_flow_lte_qci = GetNumericValue(NULL, "QoS Class Identifier (QCI) 0-70: ", g_enterActionZero, 0, 0, 70);
}

static void GetQosFilterSpecs(unpack_qos_QosFilterSpecs_t *pQoSFilterSpecs)
{
	char szIPv4v6[MAX_FIELD_SIZE] = {0};

	pQoSFilterSpecs->ip_version = GetNumericValue(NULL, "IP family (IPv4=4 (default), IPv6=6): ", g_enterActionDefault, 4, 4, 6);
	pQoSFilterSpecs->ipv4_valid_params = GetNumericValue64(NULL, "valid IPv4 parameters mask (No parameters=0 IPv4 source address=1 IPv4 destination address=2 IPv4 type of service=4): ", g_enterActionZero, 0, 0, 4);
	GetIPFromUser("IPv4 address", szIPv4v6, &pQoSFilterSpecs->ipv4_addr_1);
	GetIPFromUser("IPv4 subnet mask", szIPv4v6, &pQoSFilterSpecs->ipv4_subnet_mask_1);
	GetIPFromUser("IPv4-2 address", szIPv4v6, &pQoSFilterSpecs->ipv4_addr_2);
	GetIPFromUser("IPv4-2 subnet mask", szIPv4v6, &pQoSFilterSpecs->ipv4_subnet_mask_2);
	pQoSFilterSpecs->ipv4_val = GetNumericValue(NULL, "type of IPv4 service value: ", g_enterActionZero, 0, 0, 255);
	pQoSFilterSpecs->ipv4_mask = GetNumericValue(NULL, "Type of IPv4-2service mask: ", g_enterActionZero, 0, 0, 255);
	pQoSFilterSpecs->ipv6_valid_params = GetNumericValue64(NULL, "valid IPv6 parameters mask (No parameters=0 IPv6 source address=2 IPv6 destination address=2  IPv6 traffic class=4 IPv6 flow label=8): ", g_enterActionZero, 0, 0, 8);

	GetIPv6FromUser(szIPv4v6, pQoSFilterSpecs->ipv6_address_1, &pQoSFilterSpecs->ipv6_prefix_len_1);
	pQoSFilterSpecs->ipv6_prefix_len_1 -= 10;
	GetIPv6FromUser(szIPv4v6, pQoSFilterSpecs->ipv6_address_2, &pQoSFilterSpecs->ipv6_prefix_len_2);
	pQoSFilterSpecs->ipv6_prefix_len_2 -= 10;

	pQoSFilterSpecs->ipv6_val = GetNumericValue(NULL, "traffic class value-2: ", g_enterActionZero, 0, 0, 255);
	pQoSFilterSpecs->ipv6_mask = GetNumericValue(NULL, "Traffic class mask-2: ", g_enterActionZero, 0, 0, 255);
	pQoSFilterSpecs->ipv6_flow_label = GetNumericValue(NULL, "IPv6 flow label-2: ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
	char szMessage[512] = {0};
	strcpy(szMessage, " xport_protocol:\nNo transport protocol=0\nInternet Control Messaging Protocol=1\nTransmission Control Protocol=6\n");
	strcat(szMessage, "User Datagram Protocol=17\nEncapsulating Security Payload Protocol=50\nAuthentication Header Protocol=51\nInternet Control Messaging Protocol for IPV6=58");
	pQoSFilterSpecs->ipv6_xport_protocol = GetNumericValue(NULL, szMessage, g_enterActionZero, 0, 0, 58);
	pQoSFilterSpecs->port_valid_params_1 = GetNumericValue64(NULL, "valid port info mask (No parameters=0 Source port=1 Destination port=2): ", g_enterActionZero, 0, 0, 4);
	pQoSFilterSpecs->src_port_1 = GetNumericValue(NULL, "source port 1: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->src_range_1 = GetNumericValue(NULL, "source range 1: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->src_port_1 = GetNumericValue(NULL, "destination port 1: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->src_range_1 = GetNumericValue(NULL, "destination range 1: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->port_valid_params_2 = GetNumericValue64(NULL, "valid port info mask (No parameters=0 Source port=1 Destination port=2): ", g_enterActionZero, 0, 0, 4);
	pQoSFilterSpecs->src_port_2 = GetNumericValue(NULL, "source port 2: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->src_range_2 = GetNumericValue(NULL, "source range 2: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->des_port_2 = GetNumericValue(NULL, "destination port 2: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->des_range_2 = GetNumericValue(NULL, "destination range 2: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->icmp_valid_params = GetNumericValue64(NULL, "valid ICMP filter mask (No parameters=0 Message type=1 Destination Message code=2): ", g_enterActionZero, 0, 0, 4);
	pQoSFilterSpecs->icmp_type = GetNumericValue(NULL, "source port 2: ", g_enterActionZero, 0, 0, 255);
	pQoSFilterSpecs->icmp_code = GetNumericValue(NULL, "source range 2: ", g_enterActionZero, 0, 0, 255);
	pQoSFilterSpecs->ipsec_valid_params = GetNumericValue64(NULL, "valid IPSEC filter mask (No parameters=0 Security parameter index=1): ", g_enterActionZero, 0, 0, 4);
	pQoSFilterSpecs->ipsec_spi = GetNumericValue(NULL, "Security parameter index for IPSec: ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
	pQoSFilterSpecs->ipsec_valid_params_2 = GetNumericValue64(NULL, "valid IPSEC filter mask 2 (No parameters=0 Security parameter index=1): ", g_enterActionZero, 0, 0, 4);
	pQoSFilterSpecs->ipsec_spi_2 = GetNumericValue(NULL, "Security parameter index for IPSec 2: ", g_enterActionZero, 0, 0, MAX_INT32_VALUE_SIZE);
	pQoSFilterSpecs->ipsec_filter_id = GetNumericValue(NULL, "Unique identifier for each filter: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
	pQoSFilterSpecs->filter_precedence = GetNumericValue(NULL, "the order in which filters are applied: ", g_enterActionZero, 0, 0, MAX_INT16_VALUE_SIZE);
}

void qos_request_qos_ex()
{
	pack_qos_SLQSRequestQosExReq_t sRequestQosExReq;
	memset(&sRequestQosExReq, 0, sizeof(pack_qos_SLQSRequestQosExReq_t));
	pack_qos_QosFlowList_t sTxQosFlowList;
	pack_qos_QosFlow_t sTxQosFlow;
	pack_qos_QosFlowList_t sRxQosFlowList;
	pack_qos_QosFlow_t sRxQosFlow;
	pack_qos_QosFilterSpecsList_t sTxQosFilterSpecsList;
	pack_qos_QosFilterSpecs_t sTxQoSFilterSpecs;
	pack_qos_QosFilterSpecsList_t sRxQosFilterSpecsList;
	pack_qos_QosFilterSpecs_t sRxQoSFilterSpecs;

	uint32_t nTx_5g_qci = 0;
	pack_qos_TxRx5GQCI_t sTx5GQCI;
	uint32_t nRx_5g_qci = 0;
	pack_qos_TxRx5GQCI_t sRx5GQCI;
	uint32_t nTx_averaging_window = 0;
	pack_qos_TxRxQosAveragingWindow_t sTxQosAveragingWindow;
	uint32_t nRx_averaging_window = 0;
	pack_qos_TxRxQosAveragingWindow_t sRxQosAveragingWindow;

	char ch[2] = {0};
	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Tx QOS Flow Information:", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		memset(&sTxQosFlow, 0, sizeof(pack_qos_QosFlow_t));
		GetQosFlow(&sTxQosFlow);
		sTxQosFlowList.tx_qos_flow_len = 1;
		sTxQosFlowList.pQosFlow = &sTxQosFlow;
		sRequestQosExReq.pTxQosFlowList = &sTxQosFlowList;
	}

	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Rx QOS Flow Information", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		memset(&sRxQosFlow, 0, sizeof(pack_qos_QosFlow_t));
		GetQosFlow(&sRxQosFlow);
		sRxQosFlowList.tx_qos_flow_len = 1;
		sRxQosFlowList.pQosFlow = &sRxQosFlow;
		sRequestQosExReq.pRxQosFlowList = &sRxQosFlowList;
	}
    	
	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter TX QOS Filter Specs List", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		memset(&sTxQosFilterSpecsList, 0, sizeof(pack_qos_QosFilterSpecsList_t));
		sTxQosFilterSpecsList.tx_qos_filter_len = 1;
		memset(&sTxQoSFilterSpecs, 0, sizeof(pack_qos_QosFilterSpecs_t));
		GetQosFilterSpecs(&sTxQoSFilterSpecs);
		sTxQosFilterSpecsList.pQoSFilterSpecs = &sTxQoSFilterSpecs;
		sRequestQosExReq.pTxQosFilterSpecsList = &sTxQosFilterSpecsList;
	}

	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Rx QoS Filter Specs List", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		memset(&sRxQosFilterSpecsList, 0, sizeof(pack_qos_QosFilterSpecsList_t));
		sRxQosFilterSpecsList.tx_qos_filter_len = 1;
		memset(&sRxQoSFilterSpecs, 0, sizeof(pack_qos_QosFilterSpecs_t));
		GetQosFilterSpecs(&sRxQoSFilterSpecs);
		sRxQosFilterSpecsList.pQoSFilterSpecs = &sRxQoSFilterSpecs;
		sRequestQosExReq.pRxQosFilterSpecsList = &sRxQosFilterSpecsList;
	}

	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Tx 5G QCI", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		nTx_5g_qci = GetNumericValue64(NULL, "5G Tx QoS Class Identifier (QCI): ", g_enterActionZero, 0, 0, MAX_UINT32_VALUE_SIZE);
		sTx5GQCI.tx_rx_5g_qci_len = 1;
		sTx5GQCI.pTxRx_5g_qci = &nTx_5g_qci;
		sRequestQosExReq.pRx5GQCI = &sTx5GQCI;
	}

	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Rx 5G QCI", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		nRx_5g_qci = 0;
		sRx5GQCI.tx_rx_5g_qci_len = 1;
		sRx5GQCI.pTxRx_5g_qci = &nRx_5g_qci;
		sRequestQosExReq.pRx5GQCI = &sRx5GQCI;
	}

	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Tx QoS Averaging Window", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		nTx_averaging_window = GetNumericValue64(NULL, "duration in ms over which the GFBR and MFBR shall be calculated: ", g_enterActionZero, 0, 0, MAX_UINT32_VALUE_SIZE);
		sTxQosAveragingWindow.tx_averaging_window_len = 1;
		sTxQosAveragingWindow.pTxRx_averaging_window = &nTx_averaging_window;
		sRequestQosExReq.pTxQosAveragingWindow = &sTxQosAveragingWindow;
	}

	if (ENTER_KEY == GetStringFromUser(NULL, "'y' if you want to enter Rx QoS Averaging Window", g_enterActionExit, (char*)ch, 2))
		return;

	if (ch[0] == 'y')
	{
		nRx_averaging_window = 0;
		sRxQosAveragingWindow.tx_averaging_window_len = 1;
		sRxQosAveragingWindow.pTxRx_averaging_window = &nRx_averaging_window;
		sRequestQosExReq.pRxQosAveragingWindow = &sRxQosAveragingWindow;
	}
	
	unpack_qos_SLQSQosRequestQosExResp_t sQosRequestQosExResp;
	memset(&sQosRequestQosExResp, 0, sizeof(unpack_qos_SLQSQosRequestQosExResp_t));
	uint32_t                   		sQos_id;
    unpack_qos_TxRxQoSFlowParamError   sTxQoSFlowParamError;
	uint32_t arrTx_qos_flow_error[5];
	sTxQoSFlowParamError.tx_rx_qos_flow_error_len = 5;
	sTxQoSFlowParamError.pTxRx_qos_flow_error = arrTx_qos_flow_error;
    unpack_qos_TxRxQoSFlowParamError   sRxQoSFlowParamError;
	uint32_t arrRx_qos_flow_error[5];
	sRxQoSFlowParamError.tx_rx_qos_flow_error_len = 5;
	sRxQoSFlowParamError.pTxRx_qos_flow_error = arrRx_qos_flow_error;

    unpack_qos_TxRxQoSFilterParamError sTxQoSFilterParamError;
    uint32_t arrTx_qos_filter_error[2];
	sTxQoSFilterParamError.tx_rx_qos_filter_error_len = 2;
	sTxQoSFilterParamError.pTxRx_qos_filter_error = arrTx_qos_filter_error;
    unpack_qos_TxRxQoSFilterParamError sRxQoSFilterParamError;
    uint32_t arrRx_qos_filter_error[2];
	sRxQoSFilterParamError.tx_rx_qos_filter_error_len = 2;
	sRxQoSFilterParamError.pTxRx_qos_filter_error = arrRx_qos_filter_error;

	sQosRequestQosExResp.pQos_id = &sQos_id;
    sQosRequestQosExResp.pTxQoSFlowParamError = &sTxQoSFlowParamError;
    sQosRequestQosExResp.pRxQoSFlowParamError = &sRxQoSFlowParamError;
    sQosRequestQosExResp.pTxQoSFilterParamError = &sTxQoSFilterParamError;
    sQosRequestQosExResp.pRxQoSFilterParamError = &sRxQoSFilterParamError;

	int rtn = SendReceive(&s_QosService, PackQosRequestQosEx, "pack_qos_SLQSQosExRequest", &sRequestQosExReq, UnpackQosRequestQosEx, "unpack_qos_SLQSQosExRequest", &sQosRequestQosExResp);
	dlog(eLOG_DEBUG, "qos_request_qos_ex - ret: %d\n", rtn);
}

int PackQosGetQosInfo(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	int ret = pack_qos_SLQSQosGetQosInfo(pCtx, pReqBuf, pLen, (pack_qos_SLQSQosGetQosInfo_t*)pInput);
	dlog(eLOG_DEBUG, "pack_qos_SLQSQosGetQosInfo: %d\n", ret);

	return ret;
}

int UnpackQosGetQosInfo(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_qos_SLQSQosGetQosInfo_t *pQosGetQosInfo = (unpack_qos_SLQSQosGetQosInfo_t*)pOutput;
	int ret = unpack_qos_SLQSQosGetQosInfo(pResp, respLen, pOutput);
	if (ret == eQCWWAN_ERR_NONE)
		dlog(eLOG_INFO, "Flow_status: %d\n", *pQosGetQosInfo->pFlow_status);
	else			
		dlog(eLOG_ERROR, "unpack_qos_SLQSQosGetQosInfo: %d - %s\n", ret, helper_get_error_reason(ret));

	return ret;
}

int PackQosIndicationRegister(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	int ret = pack_qos_SLQSQosIndicationRegister(pCtx, pReqBuf, pLen, (pack_qos_SLQSQosIndicationRegister_t*)pInput);
	dlog(eLOG_DEBUG, "pack_qos_SLQSQosIndicationRegister: %d\n", ret);

	return ret;
}

int UnpackQosIndicationRegister(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	int ret = unpack_qos_SLQSQosIndicationRegister(pResp, respLen, pOutput);
	if (ret == eQCWWAN_ERR_NONE)
		dlog(eLOG_INFO, "unpack_qos_SLQSQosIndicationRegister: %d\n", ret);
	else
		dlog(eLOG_ERROR, "unpack_qos_SLQSQosIndicationRegister: %d\n", ret);

	return ret;
}

/*
 * Name:     qos_get_qos_info
 *
 * Purpose:  retrieve QoS Information.
 *
 * Params:   None
 *
 * Return:   None.
 *
 * Notes:    None
 */
void qos_get_qos_info()
{
	/* Receive the user input */
	uint32_t qosIdentifier = GetNumericValue(NULL, "Identifier for the QoS flow/instance that has been negotiated and that is being queried", g_enterActionExit, 1001, 0, 1000);

	/* If only <ENTER> is pressed by the user, return to main menu */
	if (1001 == qosIdentifier)
		return;

	pack_qos_SLQSQosGetQosInfo_t   sPackQosGetQosInfo;
	sPackQosGetQosInfo.qosIdentifier = qosIdentifier;
	unpack_qos_SLQSQosGetQosInfo_t sUnpackQosGetQosInfo;
	memset(&sUnpackQosGetQosInfo, 0, sizeof(unpack_qos_SLQSQosGetQosInfo_t));

	int rtn = SendReceive(&s_QosService, PackQosGetQosInfo, "pack_qos_SLQSQosGetQosInfo", &sPackQosGetQosInfo, UnpackQosGetQosInfo, "unpack_qos_SLQSQosGetQosInfo", &sUnpackQosGetQosInfo);

	dlog(eLOG_DEBUG, "qos_get_qos_info - ret: %d\n", rtn);
}

/*
 * Name:     qos_indication_register
 *
 * Purpose:  QOS indication register.
 *
 * Params:   None
 *
 * Return:   None.
 *
 * Notes:    None
 */
void qos_indication_register()
{
	/* Receive the user input */
	uint8_t Report_global_qos_flows = GetNumericValue(NULL, "Report Global QOS Flows (0: Do not report (default), 1: Report global QOS flows)", g_enterActionDefault, ENTER_KEY_PRESSED, 0, 1);
	uint8_t Suppress_report_flow_control = GetNumericValue(NULL, "Suppress Report Flow Control (0: Report flow control events (default), 1: Do not report flow control events)", g_enterActionDefault, ENTER_KEY_PRESSED, 0, 1);
	uint8_t Suppress_network_status_ind = GetNumericValue(NULL, "Suppress Network Status Indication (0: Do not suppress network status indication (default), 1: Suppress network status indication)", g_enterActionDefault, ENTER_KEY_PRESSED, 0, 1);

	pack_qos_SLQSQosIndicationRegister_t sPackIndicationRegister;
	memset(&sPackIndicationRegister, 0, sizeof(pack_qos_SLQSQosIndicationRegister_t));
	if (Report_global_qos_flows != ENTER_KEY)
		sPackIndicationRegister.pReport_global_qos_flows = &Report_global_qos_flows;
	if (Suppress_report_flow_control != ENTER_KEY)
		sPackIndicationRegister.pSuppress_report_flow_control = &Suppress_report_flow_control;
	if (Suppress_network_status_ind != ENTER_KEY)
		sPackIndicationRegister.pSuppress_network_status_ind = &Suppress_network_status_ind;

	unpack_qos_SLQSQosIndicationRegister_t sUnpackIndicationRegister;
	memset(&sUnpackIndicationRegister, 0, sizeof(unpack_qos_SLQSQosIndicationRegister_t));

	int rtn = SendReceive(&s_QosService, PackQosIndicationRegister, "pack_qos_SLQSQosIndicationRegister", &sPackIndicationRegister, UnpackQosIndicationRegister, "unpack_qos_SLQSQosIndicationRegister", &sUnpackIndicationRegister);

	dlog(eLOG_DEBUG, "qos_indication_register - ret: %d\n", rtn);
}

/******************************************************************************
* Option 6 : Create a profile on the device
******************************************************************************/

int packCreateProfile(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSCreateProfile(pCtx, pReqBuf, pLen, (pack_wds_SLQSCreateProfile_t*)pInput);
}

int unpackCreateProfile(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSCreateProfile(pResp, respLen, (unpack_wds_SLQSCreateProfile_t*)pOutput);
}

static int create_new_profile(
	pack_wds_SLQSCreateProfile_t *pProfileIn,
	unpack_wds_SLQSCreateProfile_t *pProfileOut)
{
	return SendReceive(&sessions[0].wdsSvc,
		packCreateProfile, "pack_wds_SLQSCreateProfile", pProfileIn,
		unpackCreateProfile, "unpack_wds_SLQSCreateProfile", pProfileOut);
}

/******************************************************************************
* Option 7 : Modify the settings of an existing profile stored on the device
******************************************************************************/

int packModifyProfile(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSModifyProfile(pCtx, pReqBuf, pLen, (pack_wds_SLQSModifyProfile_t*)pInput);
}

int unpackModifyProfile(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSModifyProfile(pResp, respLen, (unpack_wds_SLQSModifyProfile_t*)pOutput);
}

static int change_profile_setting(
	pack_wds_SLQSModifyProfile_t *pProfileIn,
	unpack_wds_SLQSModifyProfile_t *pProfileOut)
{
	return SendReceive(&sessions[0].wdsSvc,
		packModifyProfile, "pack_wds_SLQSModifyProfile", pProfileIn,
		unpackModifyProfile, "unpack_wds_SLQSModifyProfile", pProfileOut);
}

/*
 * Name:     modify_profile_settings
 *
 * Purpose:  Modify the Profile settings of the profile id selected by the user
 *           with the values entered by the user.
 *
 * Params:   None
 *
 * Return:   None
 *
 * Notes:    None
 */
void modify_profile_settings(bool bCreate)
{
	while (1)
	{
		/* Display all the profiles stored on the device */
		display_all_profiles(!bCreate);

		if (0 == indexInfo.totalProfilesOnDevice && !bCreate)
		{
			dlog(eLOG_ERROR, "No Profile exist on device for modification or check device connectivity\n\n");
			break;
		}

		uint8_t profileId = 0;
		uint8_t profileIdMatch = bCreate ? TRUE : FALSE;

		while (!profileIdMatch)
		{
			/* Prompt the user to enter the profile id whose values need to be
			 * modified.
			 */
			profileId = GetUserProfileId();

			/* If only <ENTER> is pressed by the user, return to main menu */
			if (ENTER_KEY_PRESSED == profileId)
				return;

			/* If the user has enter an invalid profile id */
			for (int idx = 0; idx < indexInfo.totalProfilesOnDevice && !profileIdMatch; idx++)
			{
				profileIdMatch = profileId == indexInfo.profileIndex[idx];
			}
		}

		struct profileInformation profileInfo;
		profileInfo.profileType = PROFILE_TYPE_UMTS;
		char                      IPAddress[MAX_FIELD_SIZE];
		uint16_t                  ProfileSize, APNSize, UserSize, PwdSize;

		pack_wds_SLQSModifyProfile_t modifyProfile;
		memset(&modifyProfile, 0, sizeof(modifyProfile));

		profileInfo.PDPType = GetPDPType();	// Get PDP Type

		if (ENTER_KEY == profileInfo.PDPType)
			return;

		modifyProfile.curProfile.SlqsProfile3GPP.pPDPtype = &profileInfo.PDPType;

		/* Prompt the user to enter the profile parameter values */
		/* Get the IP Address */
		if (ENTER_KEY != GetIPFromUser("IP", IPAddress, &profileInfo.IPAddress))
			modifyProfile.curProfile.SlqsProfile3GPP.pIPv4AddrPref = &profileInfo.IPAddress;

		/* Get the Primary DNS Address */
		if (ENTER_KEY != GetIPFromUser("PrimaryDNS Address", IPAddress, &profileInfo.primaryDNS))
			modifyProfile.curProfile.SlqsProfile3GPP.pPriDNSIPv4AddPref = &profileInfo.primaryDNS;

		/* Get the Secondary DNS Address */
		if (ENTER_KEY != GetIPFromUser("SecondaryDNS Address", IPAddress, &profileInfo.secondaryDNS))
			modifyProfile.curProfile.SlqsProfile3GPP.pSecDNSIPv4AddPref = &profileInfo.secondaryDNS;

		/* Get Authentication From the user */
		profileInfo.Authentication = GetAuthenticationValue();
		modifyProfile.curProfile.SlqsProfile3GPP.pAuthenticationPref = &profileInfo.Authentication;

		/* Get Profile Name from the user, Max size is 14 characters */
		if (ENTER_KEY != GetStringFromUser(NULL, "Profile Name", g_enterActionLeaveEmpty, (char*)profileInfo.profileName, MAX_PROFILE_NAME_SIZE))
		{
			modifyProfile.curProfile.SlqsProfile3GPP.pProfilename = profileInfo.profileName;
			ProfileSize = strlen((char*)profileInfo.profileName);
			modifyProfile.curProfile.SlqsProfile3GPP.pProfilenameSize = &ProfileSize;
		}

		/* Get APN Name from the user */
		if (ENTER_KEY != GetStringFromUser(NULL, "APN Name", g_enterActionLeaveEmpty, (char*)profileInfo.APNName, MAX_APN_SIZE))
		{
			modifyProfile.curProfile.SlqsProfile3GPP.pAPNName = profileInfo.APNName;
			APNSize = strlen((char*)profileInfo.APNName);
			modifyProfile.curProfile.SlqsProfile3GPP.pAPNnameSize = &APNSize;
		}

		/* Get User Name from the user */
		if (ENTER_KEY != GetStringFromUser(NULL, "User Name", g_enterActionLeaveEmpty, (char*)profileInfo.userName, MAX_USER_NAME_SIZE))
		{
			modifyProfile.curProfile.SlqsProfile3GPP.pUsername = profileInfo.userName;
			UserSize = strlen((char*)profileInfo.userName);
			modifyProfile.curProfile.SlqsProfile3GPP.pUsernameSize = &UserSize;
		}

		/* Get Password from the user */
		if (ENTER_KEY != GetStringFromUser(NULL, "Password", g_enterActionLeaveEmpty, (char*)profileInfo.password, MAX_FIELD_SIZE))
		{
			modifyProfile.curProfile.SlqsProfile3GPP.pPassword = profileInfo.password;
			PwdSize = strlen((char*)profileInfo.password);
			modifyProfile.curProfile.SlqsProfile3GPP.pPasswordSize = &PwdSize;
		}

		int rtn = 0;

		if (bCreate)
		{
			pack_wds_SLQSCreateProfile_t createProfile;
			memset(&createProfile, 0, sizeof(createProfile));
			createProfile.pCurProfile = &modifyProfile.curProfile;
			createProfile.pProfileType = &profileInfo.profileType;

			PackCreateProfileOut         profileInfoOut;
			memset(&profileInfoOut, 0, sizeof(profileInfoOut));

			unpack_wds_SLQSCreateProfile_t createProfileOut;
			memset(&createProfileOut, 0, sizeof(createProfileOut));
			createProfileOut.pCreateProfileOut = &profileInfoOut;
			createProfileOut.pProfileID = &profileId;

			rtn = create_new_profile(&createProfile, &createProfileOut);
		}
		else
		{
			modifyProfile.pProfileId = &profileId;
			modifyProfile.pProfileType = &profileInfo.profileType;
			//modifyProfile.curProfile.SlqsProfile3GPP.pPDPtype = &profileInfo.PDPType;

			uint16_t extendedErrorCode = (uint16_t)-1;

			unpack_wds_SLQSModifyProfile_t modifyProfileOut;
			memset(&modifyProfileOut, 0, sizeof(modifyProfileOut));
			modifyProfileOut.pExtErrorCode = &extendedErrorCode;

			rtn = change_profile_setting(&modifyProfile, &modifyProfileOut);
		}

		if (rtn != eLITE_CONNECT_APP_OK)
			dlog(eLOG_ERROR, "Profile %s Failed\nFailure cause - %d\n", bCreate ? "Creation" : "Modification", rtn);
		else
			dlog(eLOG_INFO, "Profile Settings %s successfully for Profile ID: %d\n", bCreate ? "created" : "updated",profileId);
	}
}

/******************************************************************************
* Option 5 : Display the settings for a particular profile stored on the device
******************************************************************************/

int packGetProfileSettings(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSGetProfileSettings(pCtx, pReqBuf, pLen, (pack_wds_SLQSGetProfileSettings_t*)pInput);
}

int unpackGetProfileSettings(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSGetProfileSettings(pResp, respLen, (unpack_wds_SLQSGetProfileSettings_t*)pOutput);
}

static int get_profile_setting(
	uint8_t profileType,
	uint8_t profileId,
	unpack_wds_SLQSGetProfileSettings_t *pProfile)
{
	pack_wds_SLQSGetProfileSettings_t reqarg;
	reqarg.ProfileId = profileId;
	reqarg.ProfileType = profileType;

#if SHOW_INVALID_PROFILE_ERROR
	return SendReceive(&sessions[0].wdsSvc,
		packGetProfileSettings, "pack_wds_SLQSGetProfileSettings", &reqarg,
		unpackGetProfileSettings, "unpack_wds_SLQSGetProfileSettings", pProfile);
#else
	return SendReceive(&sessions[0].wdsSvc,
		packGetProfileSettings, "pack_wds_SLQSGetProfileSettings", &reqarg,
		unpackGetProfileSettings, (void*)-1, pProfile);
#endif
}

static void display_profile_header()
{
	/* Display the header */
	int w3 = -3, w5 = -5, w8 = -8, w20 = -20;
	fprintf(stderr, "\n%*s%*s%*s%*s%*s%*s%*s%*s%*s\n",
		w3, "ID", w8, "PDPType", w20, "IPAddress",
		w20, "PrimaryDNS", w20, "SecondaryDNS", w5, "Auth", w20, "ProfileName",
		w20, "APNName", w20, "UserName");
}

/*
 * Name:     display_profile_info
 *
 * Purpose:  Display the profile information for the profile index provided by
 *           the user.
 *
 * Params:   None
 *
 * Return:   None.
 *
 * Notes:    None
 */
static void display_profile_info(uint8_t profileId, bool bDisplayHeader, bool bUpdateCache)
{
	int rtn;
	unpack_wds_SLQSGetProfileSettings_t profileSettingsRsp;
	UnPackGetProfileSettingOut profileSetOut;
	uint8_t profileType = PROFILE_TYPE_UMTS;
	uint8_t PDPType = (uint8_t)-1;
	uint32_t IPAddress = (uint32_t)-1;
	char  bufIPAddress[MAX_FIELD_SIZE];
	uint32_t primaryDNS = (uint32_t)-1;
	char  bufPrimaryDNS[MAX_FIELD_SIZE];
	uint32_t secondaryDNS = (uint32_t)-1;
	char  bufSecondaryDNS[MAX_FIELD_SIZE];
	uint8_t authentication = (uint8_t)-1;
	uint8_t  profileName[MAX_PROFILE_NAME_SIZE];
	uint16_t profileNameSize;
	uint8_t  APNName[MAX_APN_SIZE];
	uint16_t APNNameSize;
	uint8_t  Username[MAX_USER_NAME_SIZE];
	uint16_t UsernameSize;
	uint16_t  extendedErrorCode = (uint16_t)-1;
	int w3 = -3, w5 = -5, w8 = -8, w20 = -20;

	memset(&profileSettingsRsp, 0, sizeof(profileSettingsRsp));
	memset(&profileSetOut, 0, sizeof(UnPackGetProfileSettingOut));
	memset(profileName, 0, MAX_PROFILE_NAME_SIZE);
	memset(APNName, 0, MAX_APN_SIZE);
	memset(Username, 0, MAX_USER_NAME_SIZE);
	profileNameSize = sizeof(profileName);
	APNNameSize = sizeof(APNName);
	UsernameSize = sizeof(Username);
	profileSettingsRsp.pProfileSettings = &profileSetOut;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pPDPtype = &PDPType;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pIPv4AddrPref = &IPAddress;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pPriDNSIPv4AddPref = &primaryDNS;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pSecDNSIPv4AddPref = &secondaryDNS;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pAuthenticationPref = &authentication;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pProfilename = profileName;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pProfilenameSize = &profileNameSize;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pAPNName = APNName;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pAPNnameSize = &APNNameSize;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pUsername = Username;
	profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pUsernameSize = &UsernameSize;
	profileSettingsRsp.pProfileSettings->pExtErrCode = &extendedErrorCode;

	rtn = get_profile_setting(profileType, profileId, &profileSettingsRsp);
	if (rtn != eLITE_CONNECT_APP_OK)
	{
		dlog(eLOG_DEBUG, "Profile retrieving Failed. Cause - %d, Error Code - %d\n\n",
			rtn, (int)extendedErrorCode);
		return;
	}

	if (profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pProfilename == NULL)
		strcpy((char*)profileName, "Unknown");
	if (profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pAPNName == NULL)
		strcpy((char*)APNName, "Unknown");
	if (profileSettingsRsp.pProfileSettings->curProfile.SlqsProfile3GPP.pUsername == NULL)
		strcpy((char*)Username, "Unknown");

	/* Reset the buffers */
	memset(bufIPAddress, 0, MAX_FIELD_SIZE);
	memset(bufPrimaryDNS, 0, MAX_FIELD_SIZE);
	memset(bufSecondaryDNS, 0, MAX_FIELD_SIZE);

	/* Convert ULONG to Dot notation for display */
	inet_ntop(AF_INET, &IPAddress, bufIPAddress, MAX_FIELD_SIZE);
	inet_ntop(AF_INET, &primaryDNS, bufPrimaryDNS, MAX_FIELD_SIZE);
	inet_ntop(AF_INET, &secondaryDNS, bufSecondaryDNS, MAX_FIELD_SIZE);

	if (bDisplayHeader)
		display_profile_header();
		
	/* Display the retrieved profile information */
	fprintf(stderr, "%*d%*u%*s%*s%*s%*u%*s%*s%*s\n",
		w3, profileId, w8, PDPType, w20, bufIPAddress,
		w20, bufPrimaryDNS, w20, bufSecondaryDNS, w5, authentication,
		w20, profileName, w20, APNName, w20, Username);

	if (bUpdateCache)
	{
		/* Store the profile indexes for successfully retrieved profiles */
		indexInfo.profileIndex[indexInfo.totalProfilesOnDevice] = profileId;
		indexInfo.totalProfilesOnDevice++;
	}
}

static void display_profile()
{
	while (1)
	{
		/* Receive the user input */
		uint8_t profileId = GetUserProfileId();

		/* If only <ENTER> is pressed by the user, return to main menu */
		if (ENTER_KEY_PRESSED == profileId)
			return;

		display_profile_info(profileId, true, false);
	}
}

/*************************************************************************
* Option 4 : Display all the profiles stored on the device
*************************************************************************/

/*
 * Name:     display_all_profiles
 *
 * Purpose:  Display all the profiles stored on the device.
 *
 * Params:   None
 *
 * Return:   None
 *
 * Notes:    None
 */
static void display_all_profiles(bool bShowNoProfile)
{
	display_profile_header();

	indexInfo.totalProfilesOnDevice = 0;

	/* Retrieve the information for all the profiles loaded on the device */
	for (uint8_t profileId = MIN_PROFILES; profileId <= MAX_PROFILES; profileId++)
		display_profile_info(profileId, false, true);

	if (0 == indexInfo.totalProfilesOnDevice && bShowNoProfile)
		dlog(eLOG_ERROR, "No Profile exist on the device or check device connectivity\n\n");
}

/*************************************************************************
* Option 3 : Stop the currently active Data Session
*************************************************************************/

int packStopDataSession(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSStopDataSession(pCtx, pReqBuf, pLen, (pack_wds_SLQSStopDataSession_t*)pInput);
}

int unpackStopDataSession(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSStopDataSession(pResp, respLen, (unpack_wds_SLQSStopDataSession_t*)pOutput);
}

static int stop_data_session(pack_wds_SLQSStopDataSession_t *pSession, int nIdx)
{
	unpack_wds_SLQSStopDataSession_t resp;
	memset(&resp, 0, sizeof(unpack_wds_SLQSStopDataSession_t));

	return SendReceive(&sessions[nIdx].wdsSvc,
		packStopDataSession, "pack_wds_SLQSStopDataSession", pSession,
		unpackStopDataSession, "unpack_wds_SLQSStopDataSession", &resp);
}

/*
 * Name:     stop_current_datasession
 *
 * Purpose:  Stop the ongoing data session
 *
 * Params:   None
 *
 * Return:   None
 *
 * Notes:    None
 */
void stop_current_datasession(int session)
{
	if (!sessions[session].active || sessions[session].id == 0)
		return;

	pack_wds_SLQSStopDataSession_t stopSession;
	memset(&stopSession, 0, sizeof(stopSession));
	stopSession.psid = &sessions[session].id;

	int nIP = session % 2 == 0 ? IPV4 : IPV6;
	int rtn = stop_data_session(&stopSession, session);
	dlog(eLOG_INFO, "data session (ipv%d) %sstopped\n", nIP, rtn != eLITE_CONNECT_APP_OK ? "not " : "");

	if (rtn != eLITE_CONNECT_APP_OK)
		return;

	sessions[session].active = false;
	sessions[session].id = 0;

	if (sessions[session].uIPCount > 0)
		sessions[session].uIPCount--;

	if (strlen(setrouteTable[session][sessions[session].uIPCount].dst) > 0)
		SetRoute(nIP == IPV4 ? AF_INET : AF_INET6,
			setrouteTable[session][sessions[session].uIPCount].szif,
			false,
			setrouteTable[session][sessions[session].uIPCount].dst,
			nIP == IPV4 ? nMaskPrefLenV4 : nMaskPrefLenV6,
			setrouteTable[session][sessions[session].uIPCount].gw);

	if (g_ip_auto_assign)
		SetAdaptorAddress(nIP == IPV4 ? AF_INET : AF_INET6, setrouteTable[session][sessions[session].uIPCount].szif,
			false, sessions[session].ip, sessions[session].prefixlen);


	if (g_vlan_added)
	{
		const char* vlanName = GetMpdnIfName(session);

		DownAdaptorInterface(vlanName);
		DeleteVlan(vlanName);

		if (!IsAdaptorUp(VLAN_0_NAME) &&
			!IsAdaptorUp(VLAN_1_NAME) &&
			!IsAdaptorUp(VLAN_2_NAME) &&
			!IsAdaptorUp(VLAN_3_NAME))
		{
			g_vlan_added = false;
		}
	}
	else if (g_qmimux_added)
	{
		DownAdaptorInterfaces(szEthName, 2);

		g_qmimux_added = false;
	}
	else
		DownAdaptorInterfaces(szEthName, 0);

	g_default_dev = szEthName;
	g_nMtu = 0xFFFFFFFF;
	g_nMtuV0 = 0xFFFFFFFF;
	g_nMtuV1 = 0xFFFFFFFF;

	bool activeSession = false;
	for (int i = 0; i < NUM_WDS_SVC; i++)
	{
		if (sessions[i].active) activeSession = true;
	}
	if (!activeSession) g_connection_state = NONE;
}

void stop_all_datasessions()
{
	for (int i = 0; i < NUM_WDS_SVC; i++)
	{
		stop_current_datasession(i);
	}
}

void stop_one_datasession()
{
	int session = -1;
	session = GetDataSession();
	if (session == -1)
	{
		return;
	}
	stop_current_datasession(session);
}

/*************************************************************************
 * Option 2 : Start LTE Data Session
 ************************************************************************/

int packStartDataSession(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSStartDataSession(pCtx, pReqBuf, pLen, (pack_wds_SLQSStartDataSession_t*)pInput);
}

int unpackStartDataSession(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSStartDataSession(pResp, respLen, (unpack_wds_SLQSStartDataSession_t*)pOutput);
}

static int start_data_session(
	pack_wds_SLQSStartDataSession_t *pSession,
	unpack_wds_SLQSStartDataSession_t *pSessionOut,
	int nIdx)
{
	return SendReceive(&sessions[nIdx].wdsSvc,
		packStartDataSession, "pack_wds_SLQSStartDataSession", pSession,
		unpackStartDataSession, "unpack_wds_SLQSStartDataSession", pSessionOut);
}

int packStartDataSessionExp(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSStartDataSessionExp(pCtx, pReqBuf, pLen, (pack_wds_SLQSStartDataSessionExp_t*)pInput);
}

static int start_data_session_exp(
	pack_wds_SLQSStartDataSessionExp_t *pSession,
	unpack_wds_SLQSStartDataSession_t *pSessionOut,
	int nIdx)
{
	return SendReceive(&sessions[nIdx].wdsSvc,
		packStartDataSessionExp, "pack_wds_SLQSStartDataSessionExp", pSession,
		unpackStartDataSession, "unpack_wds_SLQSStartDataSession", pSessionOut);
}

int packSetIPFamilyPreference(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SLQSSetIPFamilyPreference(pCtx, pReqBuf, pLen, (pack_wds_SLQSSetIPFamilyPreference_t*)pInput);
}

int unpackSetIPFamilyPreference(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SLQSSetIPFamilyPreference(pResp, respLen, (unpack_wds_SLQSSetIPFamilyPreference_t*)pOutput);
}

int packSetMuxID(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wds_SetMuxID(pCtx, pReqBuf, pLen, (pack_wds_SetMuxID_t*)pInput);
}

int unpackSetMuxID(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	return unpack_wds_SetMuxID(pResp, respLen, (unpack_wds_SetMuxID_t*)pOutput);
}

int UpdateIPRuntimeSettings(unpack_wds_SLQSGetRuntimeSettings_t *runtime, int nIdx, uint8_t isMultiPdn)
{
	if (runtime == NULL)
	{
		dlog(eLOG_ERROR, "%s runtime data structure is NULL\n", __FUNCTION__);
		return 0;
	}

	swi_uint256_print_mask(runtime->ParamPresenceMask);

	const char* szIf = isMultiPdn == 0 ? szEthName : GetMpdnIfName(nIdx);
	char szFullIf[MAX_IFLEN*2+1];
	if (isMultiPdn == 0)
		strcpy(szFullIf, szEthName);
	else
		sprintf(szFullIf, "%s:%s", szEthName, szIf);

	int domain = AF_INET;
	char szGWAddress[INET6_ADDRSTRLEN] = { 0 };

	if (nIdx % 2 == 0)
	{
		// IPv4
		if (!swi_uint256_get_bit(runtime->ParamPresenceMask, 0x1E) ||	// IPv4
			!swi_uint256_get_bit(runtime->ParamPresenceMask, 0x20) ||	// IPv4 Gateway Address
			!swi_uint256_get_bit(runtime->ParamPresenceMask, 0x21) ||	// IPv4 Subnet Mask
			!swi_uint256_get_bit(runtime->ParamPresenceMask, 0x29))		// MTU
		{
			dlog(eLOG_WARN, "%s(%d): No runtime IPv4 Info\n", __FUNCTION__, nIdx);
			return 0;
		}

		struct in_addr ip_addr;
		ip_addr.s_addr = htonl(runtime->IPv4);
		inet_ntop(AF_INET, &ip_addr, sessions[nIdx].ip, INET6_ADDRSTRLEN);

		struct in_addr gw_addr;
		gw_addr.s_addr = htonl(runtime->GWAddressV4);
		inet_ntop(AF_INET, &gw_addr, szGWAddress, sizeof(szGWAddress));

		sessions[nIdx].prefixlen = Mask2PrefixLenV4(runtime->SubnetMaskV4);
		sessions[nIdx].uIPCount++;
	}
	else
	{
		struct in6_addr ipv6;
		domain = AF_INET6;

		// IPv6
		if (inet_ntop(AF_INET6, htonl6(runtime->IPV6AddrInfo.IPAddressV6, &ipv6), sessions[nIdx].ip, INET6_ADDRSTRLEN))
			sessions[nIdx].uIPCount++;

		inet_ntop(AF_INET6, htonl6(runtime->IPV6GWAddrInfo.gwAddressV6, &ipv6), szGWAddress, sizeof(szGWAddress));
		sessions[nIdx].prefixlen = runtime->IPV6AddrInfo.IPV6PrefixLen;
	}

	/* command to set mtu size */
	if (g_mtu_auto_update_enable && (runtime->Mtu != 0xFFFFFFFF) && (runtime->Mtu != 68)) // minimum mtu size is 68
	{
		if (isMultiPdn)
		{
			// MPDN
			if (nIdx < 2)
			{
				// vlan.0
				if (g_nMtuV0 == 0xFFFFFFFF || runtime->Mtu < g_nMtuV0)
				{
					SetAdaptorMtu(szEthName, runtime->Mtu);
					SetAdaptorMtu(szIf, runtime->Mtu);
					g_nMtu = runtime->Mtu;
					g_nMtuV0 = runtime->Mtu;
				}
			}
			else
			{
				// vlan.1
				if (g_nMtuV1 == 0xFFFFFFFF || runtime->Mtu < g_nMtuV1)
				{
					g_nMtuV1 = runtime->Mtu;

					unsigned int nMtuMax = g_nMtuV0 >= g_nMtuV1 ? g_nMtuV0 : g_nMtuV1;

					if (g_nMtu != nMtuMax)
					{
						g_nMtu = nMtuMax;
						SetAdaptorMtu(szEthName, g_nMtu);
					}

					SetAdaptorMtu(szIf, g_nMtuV1);
				}
			}
		}
		else
		{
			if (g_nMtu == 0xFFFFFFFF || runtime->Mtu < g_nMtu)
			{
				SetAdaptorMtu(szIf, runtime->Mtu);
				g_nMtu = runtime->Mtu;
			}
		}
	}

	/* set ip address to driver */
	if (g_ip_auto_assign)
	{
		SetAdaptorAddress(domain, szFullIf, true, sessions[nIdx].ip, sessions[nIdx].prefixlen);//isMultiPdn ? 0 : sessions[nIdx].mask);

		/* after setting the ip address, network adapter need sometime to become ready */
		int timeout = 0;
		while ((IsAdaptorUp(szEthName) == 0 || (isMultiPdn && IsAdaptorUp(szIf) == 0)) && timeout++ < 3)
			sleep(1);
	}

	if (sessions[nIdx].uIPCount > 0 && sessions[nIdx].uIPCount <= MAX_QMAP_INSTANCE)
	{
		StrCpy(setrouteTable[nIdx][sessions[nIdx].uIPCount - 1].szif, szIf);
		if (domain != AF_INET)
			StrCpy(setrouteTable[nIdx][sessions[nIdx].uIPCount - 1].gw, isMultiPdn ? sessions[nIdx].ip : szGWAddress);

		if (strlen(setrouteTable[nIdx][sessions[nIdx].uIPCount - 1].dst) > 0)
			SetRoute(domain, szIf, true, setrouteTable[nIdx][sessions[nIdx].uIPCount - 1].dst,
				domain == AF_INET ? nMaskPrefLenV4 : nMaskPrefLenV6,
				setrouteTable[nIdx][sessions[nIdx].uIPCount - 1].gw);
	}

	if (strlen(pingTable[nIdx][sessions[nIdx].uIPCount - 1]) > 0 && g_auto_test_enable)
		ping(domain == AF_INET ? IPV4 : IPV6, pingTable[nIdx][sessions[nIdx].uIPCount - 1], PING_COUNT, PING_DELAY);

	return 1;
}

void DisplayRuntimeSettings(int nIdx, uint8_t isMultiPdn)
{
	pack_wds_SLQSGetRuntimeSettings_t runtimeSettings;
	uint32_t reqSettings = 0xA304;	// Get IP Family/MTU/IP Address/PDP type
	memset(&runtimeSettings, 0, sizeof(pack_wds_SLQSGetRuntimeSettings_t));
	runtimeSettings.pReqSettings = &reqSettings;

	unpack_wds_SLQSGetRuntimeSettings_t runtimeSettingsOut;
	memset(&runtimeSettingsOut, 0, sizeof(unpack_wds_SLQSGetRuntimeSettings_t));

	int rtn = get_runtime_settings(&runtimeSettings, &runtimeSettingsOut, nIdx);
	if (rtn != eLITE_CONNECT_APP_OK)
	{
		dlog(eLOG_DEBUG, "Failed to get runtime settings\n");
		update_user_display(eIP_ADDRESS, "UNAVAILABLE");
	}
	else
	{
		char  bufIPAddress[INET6_ADDRSTRLEN];
		memset(bufIPAddress, 0, INET6_ADDRSTRLEN);

		if (nIdx % 2 == 0)
		{
			if (runtimeSettingsOut.IPv4 == 0)
				return;

			inet_ntop(AF_INET, &runtimeSettingsOut.IPv4, bufIPAddress, INET6_ADDRSTRLEN);
		}
		else
		{
			if (memcmp(&runtimeSettingsOut.IPV6AddrInfo.IPAddressV6, bufIPAddress, sizeof(runtimeSettingsOut.IPV6AddrInfo.IPAddressV6)) == 0)
				return;

			struct in6_addr ipv6;
			inet_ntop(AF_INET6, htonl6(runtimeSettingsOut.IPV6AddrInfo.IPAddressV6, &ipv6), bufIPAddress, INET6_ADDRSTRLEN);
		}

		update_user_display(eIP_ADDRESS, bufIPAddress);
	}

	if (rtn == eLITE_CONNECT_APP_OK)
	{
		/* Update the runtime settings field of the user window */
		UpdateIPRuntimeSettings(&runtimeSettingsOut, nIdx, isMultiPdn);
	}
}

/*
 * Name:     DisplaySessionState
 *
 * Purpose:  Display the information about the connected device
 *
 * Params:   None
 *
 * Return:   None
 *
 * Notes:    None
 */
void DisplaySessionState(int nIdx, uint8_t isMultiPdn)
{
	unpack_wds_GetSessionState_t sessionState;
	memset(&sessionState, 0, sizeof(unpack_wds_GetSessionState_t));

	int rtn = get_session_state(&sessionState, nIdx);
	if (rtn != eLITE_CONNECT_APP_OK)
	{
		dlog(eLOG_ERROR, "Failed to get session state (%d:ipv%d)\n", nIdx, nIdx % 2 == 0 ? IPV4 : IPV6);
		update_user_display(eSESSION_STATE, "UNAVAILABLE");
	}
	else
	{
		char* pStatus = "unknown";
		switch (sessionState.connectionStatus)
		{
		case 1:
			pStatus = "Disconnected";
			break;
		case 2:
			pStatus = "Connected";
			break;
		case 3:
			pStatus = "Suspended";
			break;
		case 4:
			pStatus = "Authenticating";
			break;
		default:
			break;
		}

		printf("SessionStatus (%d:ipv%d): %s\n", nIdx, nIdx % 2 == 0 ? IPV4 : IPV6, pStatus);
		update_user_display(eSESSION_STATE, pStatus);
	}

	if (sessionState.connectionStatus == 2)	// Connected
		DisplayRuntimeSettings(nIdx, isMultiPdn);
}

int packSetDataFormat(pack_qmi_t* pCtx, uint8_t* pReqBuf, uint16_t* pLen, void* pInput)
{
	return pack_wda_SetDataFormat(pCtx, pReqBuf, pLen, (pack_wda_SetDataFormat_t*)pInput);
}

int unpackSetDataFormat(uint8_t* pResp, uint16_t respLen, void* pOutput)
{
	unpack_wda_SetDataFormat_t* pDataFormatOut = (unpack_wda_SetDataFormat_t*)pOutput;
	return unpack_wda_SetDataFormat(pResp, respLen, pDataFormatOut);
}

static void PrintValues8(const char* name, uint8_t* in, uint8_t* out, bool newline)
{
	printf("  %s: ", name);

	if (in)
	{
		if (out)
			printf("%d -> %d", *in, *out);
		else
			printf("%d -> unknown", *in);
	}
	else
	{
		if (out)
			printf("not set -> %d", *out);
		else
			printf("not set -> unknown");
	}

	if (newline)
		printf("\n");
}

static void PrintValues32(const char* name, uint32_t* in, uint32_t* out, bool newline)
{
	printf("  %s: ", name);

	if (in)
	{
		if (out)
			printf("%d -> %d", *in, *out);
		else
			printf("%d -> unknown", *in);
	}
	else
	{
		if (out)
			printf("not set -> %d", *out);
		else
			printf("not set -> unknown");
	}

	if (newline)
		printf("\n");
}

/*
 * Name:     enable_qmap
 *
 * Purpose:  Enable QMAP
 *
 * Params:   bEnable - TRUE to enable QMAP, FALSE to disable
 *
 * Return:   0 - success, other values - failure
 *
 * Notes:    To enable the QMAP, use -Q option on command line.  
 *           QMAP settings should match those used by the PCIe device driver.
 */
static int enable_qmap(bool bEnable)
{
#define QOS_HDR_NOT_PRESENT	0
#define QOS_HDR_PRESENT	1
#define LINK_PROTOCOL_IP_MODE	2
#define DATA_AGG_PROTOCOL_QMAP_DISABLED	0
#define DATA_AGG_PROTOCOL_QMAP_ENABLED	5
#define MAX_DL_PKT_AGGR  10
#define MAX_UL_PKT_AGGR  10
#define QMAP_RX_BUFFER_SIZE 0x4000
#define QMAP_TX_BUFFER_SIZE 0x4000
#define EP_TYPE_HSUSB	2
#define EP_TYPE_PCIE	3
#define FLOW_CTL_NO_TE	0

	int ret = CtlService_InitializeRegularService(&s_CtlService, &s_WdaService, eWDA, NULL, NULL);
	if (ret != SUCCESS)
	{
		dlog(eLOG_ERROR, "InitializeRegularService eWDA failed.");
		return ret;
	}

	uint8_t  ReqQosHrPres = QOS_HDR_NOT_PRESENT;
	uint32_t ReqLinkProtocol = LINK_PROTOCOL_IP_MODE;
	uint32_t ReqULProtocol = bEnable ? DATA_AGG_PROTOCOL_QMAP_ENABLED : DATA_AGG_PROTOCOL_QMAP_DISABLED;
	uint32_t ReqDLProtocol = bEnable ? DATA_AGG_PROTOCOL_QMAP_ENABLED : DATA_AGG_PROTOCOL_QMAP_DISABLED;
	uint32_t ReqUL_Data_Aggregation_Max_Datagrams = MAX_UL_PKT_AGGR;
	uint32_t ReqDL_Data_Aggregation_Max_Datagrams = MAX_DL_PKT_AGGR;
	uint32_t ReqUL_Data_Aggregation_Max_Size = QMAP_TX_BUFFER_SIZE;
	uint32_t ReqDL_Data_Aggregation_Max_Size = QMAP_RX_BUFFER_SIZE;
	uint32_t ReqEP_Type = g_rmnet_if_id != RMNET_IF_ID_UNSET ? EP_TYPE_HSUSB : EP_TYPE_PCIE;
	uint32_t ReqInterface_ID = g_rmnet_if_id != RMNET_IF_ID_UNSET ? g_rmnet_if_id : 4 /* ??? */;
	uint32_t ReqDL_Padding = 0;
	uint8_t  ReqFlow_Control = FLOW_CTL_NO_TE;

	pack_wda_SetDataFormat_t dataFormat;
	memset(&dataFormat, 0, sizeof(pack_wda_SetDataFormat_t));
	dataFormat.pQlink_prot = &ReqLinkProtocol;
	dataFormat.pDl_data_aggregation_protocol = &ReqDLProtocol;
	dataFormat.pUl_data_aggregation_protocol = &ReqULProtocol;

	if (bEnable)
	{
		dataFormat.pQos_format = &ReqQosHrPres;
		dataFormat.pUl_data_aggregation_max_datagrams = &ReqUL_Data_Aggregation_Max_Datagrams;
		dataFormat.pDl_data_aggregation_max_datagrams = &ReqDL_Data_Aggregation_Max_Datagrams;
		dataFormat.pUl_data_aggregation_max_size = &ReqUL_Data_Aggregation_Max_Size;
		dataFormat.pDl_data_aggregation_max_size = &ReqDL_Data_Aggregation_Max_Size;
		dataFormat.pEp_type = &ReqEP_Type;
		dataFormat.pIface_id = &ReqInterface_ID;
		dataFormat.pDl_minimum_padding = &ReqDL_Padding;
		dataFormat.pFlow_control = &ReqFlow_Control;
	}

	uint8_t  RespQoS_header_presence = 0;
	uint32_t RespLink_Protocol = 0;
	uint32_t RespUL_Data_Aggregation_Protocol = 0;
	uint32_t RespDL_Data_Aggregation_Protocol = 0;
	uint32_t RespNdp_Signature = 0;
	uint32_t RespDL_Data_Aggregation_Max_Datagrams = 0;
	uint32_t RespDL_Data_Aggregation_Max_Size = 0;
	uint32_t RespUL_Data_Aggregation_Max_Datagrams = 0;
	uint32_t RespUL_Data_Aggregation_Max_Size = 0;
	uint32_t RespDL_Padding = 0;
	uint8_t  RespFlow_Control = 0;

	unpack_wda_SetDataFormat_t dataFormatOut;
	memset(&dataFormatOut, 0, sizeof(unpack_wda_SetDataFormat_t));
	dataFormatOut.pQos_format = &RespQoS_header_presence;
	dataFormatOut.pQlink_prot = &RespLink_Protocol;
	dataFormatOut.pUl_data_aggregation_protocol = &RespUL_Data_Aggregation_Protocol;
	dataFormatOut.pDl_data_aggregation_protocol = &RespDL_Data_Aggregation_Protocol;
	dataFormatOut.pNdp_signature = &RespNdp_Signature;
	dataFormatOut.pDl_data_aggregation_max_datagrams = &RespDL_Data_Aggregation_Max_Datagrams;
	dataFormatOut.pDl_data_aggregation_max_size = &RespDL_Data_Aggregation_Max_Size;
	dataFormatOut.pUl_data_aggregation_max_datagrams = &RespUL_Data_Aggregation_Max_Datagrams;
	dataFormatOut.pUl_data_aggregation_max_size = &RespUL_Data_Aggregation_Max_Size;
	dataFormatOut.pDl_minimum_padding = &RespDL_Padding;
	dataFormatOut.pFlow_control = &RespFlow_Control;

	ret = SendReceive(&s_WdaService,
		packSetDataFormat, "pack_wda_SLQSSetDataFormat", &dataFormat,
		unpackSetDataFormat, "unpack_wda_SLQSSetDataFormat", &dataFormatOut);

	if (ret == eQCWWAN_ERR_NONE)
	{
		printf("\nQMap data format:\n");
		PrintValues8("QOS format", dataFormat.pQos_format, dataFormatOut.pQos_format, true);
		PrintValues32("QLink protocol", dataFormat.pQlink_prot, dataFormatOut.pQlink_prot, true);
		PrintValues32("UL format", dataFormat.pUl_data_aggregation_protocol, dataFormatOut.pUl_data_aggregation_protocol, false);
		PrintValues32("Data Aggregation Max Datagrams", dataFormat.pUl_data_aggregation_max_datagrams, dataFormatOut.pUl_data_aggregation_max_datagrams, false);
		PrintValues32("Size", dataFormat.pUl_data_aggregation_max_size, dataFormatOut.pUl_data_aggregation_max_size, true);
		PrintValues32("DL format", dataFormat.pDl_data_aggregation_protocol, dataFormatOut.pDl_data_aggregation_protocol, false);
		PrintValues32("Data Aggregation Max Datagrams", dataFormat.pDl_data_aggregation_max_datagrams, dataFormatOut.pDl_data_aggregation_max_datagrams, false);
		PrintValues32("Size", dataFormat.pDl_data_aggregation_max_size, dataFormatOut.pDl_data_aggregation_max_size, true);
		PrintValues32("NDP signature", dataFormat.pNdp_signature, dataFormatOut.pNdp_signature, true);
		PrintValues32("EP type", dataFormat.pEp_type, NULL, false);
		PrintValues32("Interface ID", dataFormat.pIface_id, NULL, true);
		PrintValues32("DL minimum padding", dataFormat.pDl_minimum_padding, dataFormatOut.pDl_minimum_padding, true);
		PrintValues8("Flow control", dataFormat.pFlow_control, dataFormatOut.pFlow_control, true);
		printf("\n");
	}
	else
		dlog(eLOG_DEBUG, "Failed to set data format\n");

	CtlService_ShutDownRegularService(&s_CtlService, &s_WdaService);

	return ret;
}

bool ConnectViaApn(pack_wds_SLQSStartDataSessionExp_t* startSession, char* APNName, uint32_t* authType, char* userName, char* password)
{
	if (GetNumericValue(NULL, "Connect via 0: existing profile (default) or 1: explicit APN", g_enterActionDefault, 0, 0, 1) == 0)
		return false;

	if (ENTER_KEY != GetStringFromUser(NULL, "APN Name", g_enterActionLeaveEmpty, APNName, MAX_APN_SIZE))
		startSession->pApnName = APNName;

	*authType = GetAuthenticationValue();

	if (*authType != 0)
	{
		startSession->pAuthenticationPreference = authType;
		
		/* Get User Name from the user */
		if (ENTER_KEY != GetStringFromUser(NULL, "User Name", g_enterActionLeaveEmpty, userName, MAX_USER_NAME_SIZE))
			startSession->pUserName = userName;

		/* Get Password from the user */
		if (ENTER_KEY != GetStringFromUser(NULL, "Password", g_enterActionLeaveEmpty, password, MAX_FIELD_SIZE))
			startSession->pPassword = password;
	}

	return startSession->pApnName != NULL;
}

/*
 * Name:     start_datasession
 *
 * Purpose:  Starts a LTE or CDMA Data Session
 *
 * Params:   isCdma - TRUE for CDMA connection
 *			 isLTE - TRUE for LTE connection
 *
 * Return:   None
 *
 * Notes:    None
 */
void start_datasession(uint8_t isCdma, uint8_t isLTE, int acProfile, int ipType)
{
	pack_wds_SLQSStartDataSession_t startSession;
	unpack_wds_SLQSStartDataSession_t startSessionOut;
	unpack_dms_GetModelID_t modelId;
	pack_wds_SLQSSetIPFamilyPreference_t setIPfamily;
	unpack_wds_SLQSSetIPFamilyPreference_t setIPfamilyOut;
	pack_nas_SLQSSetSysSelectionPrefExt_t setNetSelPref;
	unpack_nas_SLQSSetSysSelectionPrefExt_t setNetSelPrefOut;
	struct nas_netSelectionPref netSelPref;
	uint8_t                       technology = (isCdma) ? TECHNOLOGY_3GPP2 : TECHNOLOGY_3GPP;
	uint32_t                      profileId3gpp = acProfile;
	int                           rtn;
	uint32_t                      failReason = 0;
	uint8_t                       idx = 0;
	const char*					  type = 1 == isCdma ? (1 == isLTE ? "CDMA" : "RUIM") : (1 == isLTE ? "LTE" : "UMTS");

	// Reset MTU
	g_nMtu = 0xFFFFFFFF;
	g_nMtuV0 = 0xFFFFFFFF;
	g_nMtuV1 = 0xFFFFFFFF;

	for (int i = 0; i < NUM_WDS_SVC; i++)
	{
		if (sessions[i].active)
		{
			dlog(eLOG_INFO, "Data session is already established,only one instance supported now\n\n");
			return;
		}
	}

	get_model_id(&modelId);

	if (g_mode == QMUX_INTERFACE_DIRECT && g_rmnet_if_id != RMNET_IF_ID_UNSET &&
		enable_qmap(strstr(modelId.modelid, "EM75") == NULL) != SUCCESS)	// Disable QMAP for 9x50 in rmnet mode
		dlog(eLOG_ERROR, "enable_qmap failed.\n");

	/* If connected device is GOBI, return after displaying an error message
	 * as LTE data call is not supported on GOBI devices.
	 */
	memset(&modelId, 0, sizeof(modelId));

	if (1 == isLTE || 1 == isCdma)
	{
		/* If connected device is GOBI, return after displaying an error message
		 * as this data call is not supported on GOBI devices.
		 */
		if (strstr(modelId.modelid, "MC83"))
		{
			dlog(eLOG_ERROR, "%s Data call is not supported on this device!!!\n", type);
			return;
		}
	}

	memset(&netSelPref, 0, sizeof(struct nas_netSelectionPref));
	GetUserNetworkSelectionPreference(&netSelPref);

	if (netSelPref.netReg == 1 || netSelPref.netReg != g_NetSelPref)
	{
		memset(&setNetSelPref, 0, sizeof(pack_nas_SLQSSetSysSelectionPrefExt_t));
		memset(&setNetSelPrefOut, 0, sizeof(unpack_nas_SLQSSetSysSelectionPrefExt_t));
		setNetSelPref.pNetSelPref = &netSelPref;
		rtn = set_network_selection_pref(&setNetSelPref, &setNetSelPrefOut);
		if (rtn != eLITE_CONNECT_APP_OK)
		{
			dlog(eLOG_ERROR, "fail to set network selection preference. Error: %d\n", rtn);
			return;
		}

		g_NetSelPref = netSelPref.netReg;
	}

	if (1 == isLTE &&
		((0 == strcmp("SL9090", modelId.modelid) || 0 == strcmp("MC9090", modelId.modelid)) && (1 == isCdma)))
	{
		memset(&startSession, 0, sizeof(startSession));
		memset(&startSessionOut, 0, sizeof(startSessionOut));

		/* Fill the information for required data session, for SL9090, it only supports mono PDN,
		   hence, do not consider mutiple PDN in this case */
		startSession.pTech = &technology;
		startSessionOut.psid = &sessions[0].id;
		startSessionOut.pFailureReason = &failReason;
		rtn = start_data_session(&startSession, &startSessionOut, 0);

		if (rtn != eLITE_CONNECT_APP_OK)
		{
			dlog(eLOG_DEBUG, "fail to start %s data session for SL/MC9090\n"\
				"Failure cause - %d\nError Code - %u\n\n", type,
				rtn, failReason);
			return;
		}
		else
		{
			dlog(eLOG_DEBUG, "%s data session started successfully\n", type);
			sessions[0].active = true;
		}
	}
	else
	{
		int i = SPDN_CONN_START;	// Where to start in the loop
		int limit = SPDN_LIMIT;
		uint8_t IPFamilyPreference = IPv4_FAMILY_PREFERENCE;

		memset(&startSession, 0, sizeof(startSession));
		memset(&startSessionOut, 0, sizeof(startSessionOut));
		memset(&setIPfamily, 0, sizeof(setIPfamily));
		memset(&setIPfamilyOut, 0, sizeof(setIPfamilyOut));

		char APNName[MAX_APN_SIZE];
		char userName[MAX_USER_NAME_SIZE];
		char password[MAX_FIELD_SIZE];
		uint32_t authType = 0;

		bool connViaProfile = 1;
		pack_wds_SLQSStartDataSessionExp_t startSessionExp;
		memset(&startSessionExp, 0, sizeof(startSessionExp));

		if (acProfile != 0)
			IPFamilyPreference = ipType == 1 ? IPv4_FAMILY_PREFERENCE : (ipType == 2 ? IPv6_FAMILY_PREFERENCE : IPv4v6_FAMILY_PREFERENCE);
		else if (1 != isCdma || 0 != isLTE)
		{
			connViaProfile = !ConnectViaApn(&startSessionExp, APNName, &authType, userName, password);
			if (connViaProfile)
			{
				/* handle data connection for NON-SL9090 modules */
				/* Display all the profiles stored on the device */
				display_all_profiles(false);

			if (0 == indexInfo.totalProfilesOnDevice)
			{
				dlog(eLOG_WARN, "No Profiles exist on the device for Data session\n"
						"or check device connectivity\n\n");
					return;
				}

				uint8_t profileIdMatch = FALSE;

				while (!profileIdMatch)
				{
					/* Get the profile id using which the data session needs to be started */
					profileId3gpp = GetUserProfileId();

					/* If only <ENTER> is pressed by the user, return to main menu */
					if (ENTER_KEY_PRESSED == profileId3gpp)
						return;

					/* If the user has enter an invalid profile id */
					for (idx = 0; !profileIdMatch && idx < indexInfo.totalProfilesOnDevice; idx++)
					{
						profileIdMatch = profileId3gpp == indexInfo.profileIndex[idx];
					}
				}
			}

			IPFamilyPreference = GetIPFamilyPreference();
		}

			g_default_dev = szEthName;
		
		int increament = 2;	// How much to increase in the loop, default for IPv4 or IPv6 only
		if (IPFamilyPreference == IPv4v6_FAMILY_PREFERENCE)
			increament = 1;	// Increase by 1 to cover both IPv4 and IPv6
#if FIRST_CONNECTION_IPV6
		else if (IPFamilyPreference == IPv4_FAMILY_PREFERENCE)
			i--;	// Start from IPv4

		for (; i >= limit; i -= increament)	// HSWOEMP-1595 Make IPv6 connection before IPv4
#else
		else if (IPFamilyPreference == IPv6_FAMILY_PREFERENCE)
			i++;	// Start from IPv6

		for (; i < limit; i += increament)
#endif
		{
			setIPfamily.IPFamilyPreference = i % 2 == 0 ? IPv4_FAMILY_PREFERENCE : IPv6_FAMILY_PREFERENCE;

			int rtn = SendReceive(&sessions[i].wdsSvc,
				packSetIPFamilyPreference, "pack_wds_SLQSSetIPFamilyPreference", &setIPfamily,
				unpackSetIPFamilyPreference, "unpack_wds_SLQSSetIPFamilyPreference", &setIPfamilyOut);
			dlog(eLOG_INFO, "\nWDS[%d] setIPFamily preference %d returns %d\n", i, setIPfamily.IPFamilyPreference, rtn);

			if (g_vlan_added || g_qmimux_added)
			{
				// wds-bind-mux-data-port to bind the qmux ID.
				uint32_t nEpType = 2;	// HSUSB
				uint32_t nIfId = (uint32_t)g_rmnet_if_id;
				uint8_t  nMuxID = i / 2;	// 0 for VLAN ID 4094
				uint32_t nClientType = 1;	// Tethered
				pack_wds_SetMuxID_t tpack_wds_SetMuxID;
				memset(&tpack_wds_SetMuxID, 0, sizeof(tpack_wds_SetMuxID));

				if (g_qmimux_added)
				{
					nMuxID++;
					tpack_wds_SetMuxID.pEpType = &nEpType;
					tpack_wds_SetMuxID.pIfId = &nIfId;
					tpack_wds_SetMuxID.pClientType = &nClientType;
				}

				tpack_wds_SetMuxID.pMuxID = &nMuxID;
				unpack_wds_SetMuxID_t tunpack_wds_SetMuxID;
				memset(&tunpack_wds_SetMuxID, 0, sizeof(tunpack_wds_SetMuxID));

				rtn = SendReceive(&sessions[i].wdsSvc,
					packSetMuxID, "pack_wds_SetMuxID", &tpack_wds_SetMuxID,
					unpackSetMuxID, "unpack_wds_SetMuxID", &tunpack_wds_SetMuxID);
				dlog(eLOG_INFO, "WDS[%d] BindMuxID %d returns %d\n", i, nMuxID, rtn);
			}

			startSessionOut.psid = &sessions[i].id;
			startSessionOut.pFailureReason = &failReason;

			if (!connViaProfile)
				rtn = start_data_session_exp(&startSessionExp, &startSessionOut, i);
			else
			{
				if (isCdma)
				{
					if (isLTE)
					{
						profileId3gpp += CDMA_PROFILE_OFFSET;
						startSession.pprofileid3gpp2 = &profileId3gpp;
					}
					else
						startSession.pTech = &technology;
				}
				else
				{
					startSession.pprofileid3gpp = &profileId3gpp;
				}

				rtn = start_data_session(&startSession, &startSessionOut, i);
			}

			if (rtn != eLITE_CONNECT_APP_OK)
				dlog(eLOG_ERROR, "WDS[%d] failed to start %s data session. Failure cause - %d, Error Code - %u\n", i, type, rtn, failReason);
			else
			{
				if (connViaProfile)
					dlog(eLOG_INFO, "WDS[%d] %s IPv%d data session started successfully for Profile ID: %u\n", i, type, setIPfamily.IPFamilyPreference, profileId3gpp);
				else
					dlog(eLOG_INFO, "WDS[%d] %s IPv%d data session started successfully on %s\n", i, type, setIPfamily.IPFamilyPreference, APNName);

				sessions[i].active = true;

				DisplaySessionState(i, g_vlan_added || g_qmimux_added);
			}
		}
	}

	for (int i = 0; i < NUM_WDS_SVC; i++)
	{
		if (sessions[i].active)
		{
			update_user_display(eCALL_STATUS, "CONNECTED");
			break;
		}
	}

	g_connection_state = SINGLE;
}

void start_multipdn_datasession(int acProfile)
{
	pack_wds_SLQSStartDataSession_t startSession;
	unpack_wds_SLQSStartDataSession_t startSessionOut;
	unpack_dms_GetModelID_t modelId;
	pack_wds_SLQSSetIPFamilyPreference_t setIPfamily;
	unpack_wds_SLQSSetIPFamilyPreference_t setIPfamilyOut;
	pack_nas_SLQSSetSysSelectionPrefExt_t setNetSelPref;
	unpack_nas_SLQSSetSysSelectionPrefExt_t setNetSelPrefOut;
	struct nas_netSelectionPref netSelPref;
	uint32_t                      profileId3gpp = acProfile;
	uint32_t                      failReason = 0;
	uint8_t                       idx = 0;
	int							  rtn;

	// Reset MTU
	g_nMtu = 0xFFFFFFFF;
	g_nMtuV0 = 0xFFFFFFFF;
	g_nMtuV1 = 0xFFFFFFFF;

	if (g_connection_state == SINGLE)
	{
		dlog(eLOG_INFO, "Stop current data session before starting a multipdn data session\n\n");
		return;
	}

	int sessionNumber = -1;
	for (int i = 0; i < NUM_WDS_SVC; i+=2)
	{
		if (!sessions[i].active && !sessions[i+1].active)
		{
			sessionNumber = i;
			break;
		}
	}

	if (sessionNumber == -1)
	{
		dlog(eLOG_INFO, "Maximum supported number of data sessions are currently active\n\n");
		return;
	}

	if (g_mode == QMUX_INTERFACE_DIRECT && g_rmnet_if_id != 0 && enable_qmap(true) != SUCCESS)
		dlog(eLOG_ERROR, "enable_qmap failed.\n");

	memset(&modelId, 0, sizeof(modelId));

	memset(&netSelPref, 0, sizeof(struct nas_netSelectionPref));
	GetUserNetworkSelectionPreference(&netSelPref);

	if (netSelPref.netReg == 1 || netSelPref.netReg != g_NetSelPref)
	{
		memset(&setNetSelPref, 0, sizeof(pack_nas_SLQSSetSysSelectionPrefExt_t));
		memset(&setNetSelPrefOut, 0, sizeof(unpack_nas_SLQSSetSysSelectionPrefExt_t));
		setNetSelPref.pNetSelPref = &netSelPref;
		rtn = set_network_selection_pref(&setNetSelPref, &setNetSelPrefOut);
		if (rtn != eLITE_CONNECT_APP_OK)
		{
			dlog(eLOG_ERROR, "fail to set network selection preference. Error: %d\n", rtn);
			return;
		}

		g_NetSelPref = netSelPref.netReg;
	}

	uint8_t IPFamilyPreference = IPv4_FAMILY_PREFERENCE;

	memset(&startSession, 0, sizeof(startSession));
	memset(&startSessionOut, 0, sizeof(startSessionOut));
	memset(&setIPfamily, 0, sizeof(setIPfamily));
	memset(&setIPfamilyOut, 0, sizeof(setIPfamilyOut));

	char APNName[MAX_APN_SIZE];
	char userName[MAX_USER_NAME_SIZE];
	char password[MAX_FIELD_SIZE];
	uint32_t authType = 0;

	bool connViaProfile = 1;
	pack_wds_SLQSStartDataSessionExp_t startSessionExp;
	memset(&startSessionExp, 0, sizeof(startSessionExp));

	connViaProfile = !ConnectViaApn(&startSessionExp, APNName, &authType, userName, password);
	if (connViaProfile)
	{

		/* handle data connection for NON-SL9090 modules */
		/* Display all the profiles stored on the device */
		display_all_profiles(false);

		uint8_t profileIdMatch = FALSE;

		while (!profileIdMatch)
		{
			/* Get the profile id using which the data session needs to be started */
			profileId3gpp = GetUserProfileId();

			/* If only <ENTER> is pressed by the user, return to main menu */
			if (ENTER_KEY_PRESSED == profileId3gpp)
				return;

			/* If the user has enter an invalid profile id */
			for (idx = 0; !profileIdMatch && idx < indexInfo.totalProfilesOnDevice; idx++)
			{
				profileIdMatch = profileId3gpp == indexInfo.profileIndex[idx];

			}
		}
	}

	IPFamilyPreference = GetIPFamilyPreference();
	
	if (g_rmnet_if_id != RMNET_IF_ID_UNSET)
	{
		if (AddQmiMuxIf(szEthName, QMIMUX_0_ID) >= 0 &&
			AddQmiMuxIf(szEthName, QMIMUX_1_ID) >= 0 && 
			AddQmiMuxIf(szEthName, QMIMUX_2_ID) >= 0 && 
			AddQmiMuxIf(szEthName, QMIMUX_3_ID) >= 0)
			g_qmimux_added = true;
		if (!g_qmimux_added)
		{
			dlog(eLOG_ERROR, "Unable to add QmiMux for multiple PDN\n\n");
			return;
		}

		g_default_dev = QMIMUX_0_NAME;
	}
	else 
	{
		if (AddVlan(szEthName, VLAN_0_NAME, VLAN_0_ID) >= 0 &&
			AddVlan(szEthName, VLAN_1_NAME, VLAN_1_ID) >= 0 && 
			AddVlan(szEthName, VLAN_2_NAME, VLAN_2_ID) >= 0 && 
			AddVlan(szEthName, VLAN_3_NAME, VLAN_3_ID) >= 0)
			g_vlan_added = true;
		if (!g_vlan_added)
		{
			dlog(eLOG_ERROR, "Unable to add VLAN for multiple PDN\n\n");
			return;
		}

		g_default_dev = VLAN_0_NAME;
	}

	int i = sessionNumber + 1;	// Where to start in the loop
	int limit = i-1;
	int increment = 2;	// How much to increase in the loop, default for IPv4 or IPv6 only
	if (IPFamilyPreference == IPv4v6_FAMILY_PREFERENCE)
		increment = 1;	// Increase by 1 to cover both IPv4 and IPv6
#if FIRST_CONNECTION_IPV6
	else if (IPFamilyPreference == IPv4_FAMILY_PREFERENCE)
		i--;	// Start from IPv4

	for (; i >= limit; i -= increment)	// HSWOEMP-1595 Make IPv6 connection before IPv4
#else
	else if (IPFamilyPreference == IPv6_FAMILY_PREFERENCE)
		i++;	// Start from IPv6

	for (; i < limit; i += increament)
#endif
	{
		setIPfamily.IPFamilyPreference = i % 2 == 0 ? IPv4_FAMILY_PREFERENCE : IPv6_FAMILY_PREFERENCE;

		rtn = SendReceive(&sessions[i].wdsSvc,
			packSetIPFamilyPreference, "pack_wds_SLQSSetIPFamilyPreference", &setIPfamily,
			unpackSetIPFamilyPreference, "unpack_wds_SLQSSetIPFamilyPreference", &setIPfamilyOut);
		dlog(eLOG_INFO, "\nWDS[%d] setIPFamily preference %d returns %d\n", i, setIPfamily.IPFamilyPreference, rtn);


		// wds-bind-mux-data-port to bind the qmux ID.
		uint32_t nEpType = 2;	// HSUSB
		uint32_t nIfId = (uint32_t)g_rmnet_if_id;
		uint8_t  nMuxID = sessionNumber/2;	// 0 for VLAN ID 4094
		uint32_t nClientType = 1;	// Tethered
		pack_wds_SetMuxID_t tpack_wds_SetMuxID;
		memset(&tpack_wds_SetMuxID, 0, sizeof(tpack_wds_SetMuxID));

		if (g_qmimux_added)
		{
			nMuxID++;
			tpack_wds_SetMuxID.pEpType = &nEpType;
			tpack_wds_SetMuxID.pIfId = &nIfId;
			tpack_wds_SetMuxID.pClientType = &nClientType;
		}

		tpack_wds_SetMuxID.pMuxID = &nMuxID;
		unpack_wds_SetMuxID_t tunpack_wds_SetMuxID;
		memset(&tunpack_wds_SetMuxID, 0, sizeof(tunpack_wds_SetMuxID));

		rtn = SendReceive(&sessions[i].wdsSvc,
			packSetMuxID, "pack_wds_SetMuxID", &tpack_wds_SetMuxID,
			unpackSetMuxID, "unpack_wds_SetMuxID", &tunpack_wds_SetMuxID);
		dlog(eLOG_INFO, "WDS[%d] BindMuxID %d returns %d\n", i, nMuxID, rtn);
		
		startSessionOut.psid = &sessions[i].id;
		startSessionOut.pFailureReason = &failReason;

		if (!connViaProfile)
			rtn = start_data_session_exp(&startSessionExp, &startSessionOut, i);
		else
		{
			startSession.pprofileid3gpp = &profileId3gpp;

			rtn = start_data_session(&startSession, &startSessionOut, i);
		}

		if (rtn != eLITE_CONNECT_APP_OK)
			dlog(eLOG_ERROR, "WDS[%d] failed to start data session. Failure cause - %d, Error Code - %u\n", i, rtn, failReason);
		else
		{
			if (connViaProfile)
				dlog(eLOG_INFO, "WDS[%d] IPv%d data session started successfully for Profile ID: %u\n", i, setIPfamily.IPFamilyPreference, profileId3gpp);
			else
				dlog(eLOG_INFO, "WDS[%d] IPv%d data session started successfully on %s\n", i, setIPfamily.IPFamilyPreference, APNName);

			sessions[i].active = true;

			DisplaySessionState(i, g_vlan_added || g_qmimux_added);
		}
	}

	for (int i = 0; i < NUM_WDS_SVC; i++)
	{
		if (sessions[i].active)
		{
			update_user_display(eCALL_STATUS, "CONNECTED");
			break;
		}
	}

	g_connection_state = MULTIPDN;
}

/*
 * Name:     display_device_info
 *
 * Purpose:  Display the information about the connected device
 *
 * Params:   None
 *
 * Return:   None
 *
 * Notes:    None
 */
static void display_device_info()
{
	int rtn;
	unpack_dms_GetModelID_t modelId;
	unpack_nas_GetHomeNetwork_t homeNw;
	unpack_nas_SLQSGetSysSelectionPrefExt_t netSelPref;
	uint8_t NetSelPref = 0;
	uint16_t ModePref = 0xFFFF;
	uint32_t GWAcqPref = 0xFFFFFFFF;
	char  *pNAString = "UNAVAILABLE";

	/* Get the Device Model ID */
	memset(&modelId, 0, sizeof(unpack_dms_GetModelID_t));

	rtn = get_model_id(&modelId);
	if (rtn != eLITE_CONNECT_APP_OK)
	{
		dlog(eLOG_DEBUG, "Failed to get model ID\n");
		update_user_display(eMODEL_ID, pNAString);
	}
	else
	{
		/* Update the model ID field of the user window */
		update_user_display(eMODEL_ID, modelId.modelid);
	}

	/* Get the Home Network */
	memset(&homeNw, 0, sizeof(unpack_nas_GetHomeNetwork_t));

	rtn = get_home_network(&homeNw);
	if (rtn != eLITE_CONNECT_APP_OK)
	{
		dlog(eLOG_DEBUG, "Failed to get home network\n");
		update_user_display(eHOME_NETWORK, pNAString);
	}
	else
	{
		/* Update the model ID field of the user window */
		update_user_display(eHOME_NETWORK, homeNw.name);
	}

	/* Get Network selection preference */
	memset(&netSelPref, 0, sizeof(unpack_nas_SLQSGetSysSelectionPrefExt_t));
	netSelPref.pNetSelPref = &NetSelPref;
	netSelPref.pModePref = &ModePref;
	netSelPref.pGWAcqOrderPref = &GWAcqPref;

	rtn = get_network_selection_pref(&netSelPref);
	if (rtn != eLITE_CONNECT_APP_OK)
	{
		dlog(eLOG_DEBUG, "Failed to get network selection preference\n");
		update_user_display(eNETWORK_PREF, pNAString);
	}
	else
	{
		/* Update the network selection preference field of the user window */
		update_user_display(eNETWORK_PREF, NetSelPref == 0 ? "auto" : "manual");
		g_NetSelPref = NetSelPref;
	}

	for (int i = 0; i < NUM_WDS_SVC; i++)
		DisplaySessionState(i, 0);
}

static void run_tests(int acProfile, int ipType, bool bTmMon, bool bTsMon)
{
	/* Initialize the output log file */
	initialize_display();

	/* Being here means, device is connected, update the required field */
	update_user_display(eDEVICE_STATE, "DEVICE CONNECTED");

	/* Display the information about the connected device */
	display_device_info();

	/* Register thermal mitigation and thermal sensor indications */
	if (bTmMon)
		register_mit_ind(true);
	if (bTsMon)
		register_ts_ind(true);

	SetTermHandler(DEFAULT_TERMHANDLER, NULL);

	bool bExiting = false;
	bool bAC = acProfile != 0;
	char selOption[OPTION_LEN];
	char *pEndOfLine = NULL;
	unsigned int len = 0;
	uint32_t userOption;

	while (!bExiting)
	{
		if (acProfile)
			userOption = eSTART_LTE_DATA_SESSION;
		else
		{
			/* Print the menu */
#ifdef ALL_CONNECTION_TYPES
			fprintf(stderr, "\nPlease select one of the following options or press <Enter> to exit:\n"\
				"%d.\tStart UMTS data session\n"\
				"%d.\tStart LTE data session\n"\
				"%d.\tStart CDMA data session\n"\
				"%d.\tStart RUIM data session\n"\
				"%d.\tStart Multiple PDN data session\n"\
				"%d.\tStop a currently active data session\n"\
				"%d.\tStop all currently active data sessions\n"\
				"%d.\tDisplay all the profiles stored on the device\n"\
				"%d.\tDisplay the settings for a particular profile stored on the device\n"\
				"%d.\tCreate a profile on the device\n"\
				"%d.\tModify the settings of an existing profile stored on the device\n"\
				"%d.\tDelete a profile stored on the device\n"\
				"%d.\tScan available networks\n"\
				"%d.\tEnable QOS Event\n"\
				"%d.\tDisable QOS Event\n"\
				"%d.\tGet QOS Information\n"\
				"%d.\tGet Packet Statistics\n"\
				"%d.\tGet Current Channel Rate\n"\
				"Option : ",
				eSTART_UMTS_DATA_SESSION,
				eSTART_LTE_DATA_SESSION,
				eSTART_CDMA_DATA_SESSION,
				eSTART_CDMA_RUIM_DATASESSION,
				eSTART_PDN_DATASESSION,
				eSTOP_DATA_SESSION,
				eSTOP_DATA_SESSIONS,
				eDISPLAY_ALL_PROFILES,
				eDISPLAY_SINGLE_PROFILE,
				eCREATE_PROFILE,
				eMODIFY_PROFILE_SETTINGS,
				eDELETE_PROFILE
				eSCAN_NETWORKS
				eEnable_QOS_EVENT,
				eDisable_QOS_EVENT,
				eREQUEST_QOSEX,
				eGET_QOSINFO,
				eSet_QOSIndicationRegister,
				eGET_PKT_STATS,
				eGET_CURRENT_CHANNEL_RATE);
#else
			fprintf(stderr, "\nPlease select one of the following options or press <Enter> to exit:\n"\
				"%d.\tStart data session\n"\
				"%d.\tStart Multiple PDN data session\n"\
				"%d.\tStop a currently active data session\n"\
				"%d.\tStop all currently active data sessions\n"\
				"%d.\tDisplay all the profiles stored on the device\n"\
				"%d.\tDisplay the settings for a particular profile stored on the device\n"\
				"%d.\tCreate a profile on the device\n"\
				"%d.\tModify the settings of an existing profile stored on the device\n"\
				"%d.\tDelete a profile stored on the device\n"\
				"%d.\tScan available networks\n"\
				"%d.\tEnable QOS Event\n"\
				"%d.\tDisable QOS Event\n"\
				"%d.\tRequest QOS Expaded\n"\
				"%d.\tGet QOS Information\n"\
				"%d.\tQOS Indication Register\n"\
				"%d.\tGet Packet Statistics\n"\
				"%d.\tGet Current Channel Rate\n"\
				"Option : ",
				eSTART_LTE_DATA_SESSION,
				eSTART_PDN_DATASESSION,
				eSTOP_DATA_SESSION,
				eSTOP_DATA_SESSIONS,
				eDISPLAY_ALL_PROFILES,
				eDISPLAY_SINGLE_PROFILE,
				eCREATE_PROFILE,
				eMODIFY_PROFILE_SETTINGS,
				eDELETE_PROFILE,
				eSCAN_NETWORKS,
				eEnable_QOS_EVENT,
				eDisable_QOS_EVENT,
				eREQUEST_QOSEX,
				eGET_QOSINFO,
				eSet_QOSIndicationRegister,
				eGET_PKT_STATS,
				eGET_CURRENT_CHANNEL_RATE);
#endif

			/* Receive the input from the user */
			while (fgets(selOption, (OPTION_LEN), stdin) == NULL);

			/* If '/n' character is not read, there are more characters in input
			 * stream. Clear the input stream.
			 */
			pEndOfLine = strchr(selOption, ENTER_KEY);
			if (NULL == pEndOfLine)
			{
				FlushStdinStream();
			}

#ifdef DBG
			fprintf(stderr, "Selected Option : %s\n", selOption);
#endif

			len = strlen(selOption);

			/* If only <ENTER> is pressed by the user quit Application */
			if (ENTER_KEY == selOption[0])
			{
				bExiting = true;
				if (bAC)
					userOption = eSTOP_DATA_SESSION;
				else
					continue;
			}
			else
			{

				/* check descriptor is still valid or not */
				//ret = fcntl(wds_fd, F_GETFD);
				//if (ret < 0) {
				//	fprintf(stderr, "Device seems disconnected, exiting application!\n");
				//	break;
				//}

				/* Convert the option added by user into integer */
				selOption[len - 1] = '\0';
				userOption = atoi(selOption);
			}
		}

		/* Process user input */
		switch (userOption) {
		case eSTART_LTE_DATA_SESSION:
			start_datasession(0, 1, acProfile, ipType);
			acProfile = 0;
			break;

#ifdef ALL_CONNECTION_TYPES
		case eSTART_UMTS_DATA_SESSION:
			start_datasession(0, 0, 0, 0);
			break;

		case eSTART_CDMA_DATA_SESSION:
			start_datasession(1, 1, 0, 0);
			break;

		case eSTART_CDMA_RUIM_DATASESSION:
			start_datasession(1, 0, 0, 0);
			break;
#endif
		case eSTART_PDN_DATASESSION:
			start_multipdn_datasession(acProfile);
			break;

		case eSTOP_DATA_SESSION:
			bAC = false;
			if (g_connection_state == NONE || g_connection_state == SINGLE)
				stop_all_datasessions();
			else
				stop_one_datasession();
			break;

		case eSTOP_DATA_SESSIONS:
			bAC = false;
			stop_all_datasessions();
			break;

		case eDISPLAY_ALL_PROFILES:
			display_all_profiles(true);
			break;

		case eDISPLAY_SINGLE_PROFILE:
			display_profile();
			break;

		case eCREATE_PROFILE:
			modify_profile_settings(true);
			break;

		case eMODIFY_PROFILE_SETTINGS:
			modify_profile_settings(false);
			break;

		case eDELETE_PROFILE:
			delete_profile_from_device();
		break;

		case eSCAN_NETWORKS:
			scan_available_networks();
		break;

		case eEnable_QOS_EVENT:
			qos_enable_qos_event(true);
		break;

		case eDisable_QOS_EVENT:
			qos_enable_qos_event(false);
		break;

		case eREQUEST_QOSEX:
			qos_request_qos_ex();
		break;

		case eGET_QOSINFO:
			qos_get_qos_info();
		break;
		case eSet_QOSIndicationRegister:
			qos_indication_register();
		break;
		case eGET_PKT_STATS:
			get_pkt_stats();
		break;
		case eGET_CURRENT_CHANNEL_RATE:
			get_Current_channel_rate();
		break;

		default:
			break;
		}
	}

	SetTermHandler(NULL, NULL);

	g_exiting = true;

	if (bTmMon)
		register_mit_ind(false);
	if (bTsMon)
		register_ts_ind(false);

	return;
}

static int EnableQmiIndications(MbimTransport* pTransport)
{
	int ret = FAILURE;

	uint32_t transactionId = MbimTransport_GetNextTransactionId(pTransport);

	uint8_t responseInformationBuffer[MBIM_MAX_CTRL_TRANSFER];

	MbimSyncObject syncObject;
	MbimSyncObject_Initialize(&syncObject, responseInformationBuffer, sizeof(responseInformationBuffer));

	MbimTransaction transaction;

	MbimTransaction_Initialize(
		&transaction,
		transactionId,
		MbimSyncObject_DoneCallback,
		&syncObject
	);

	MbimEventEntry elements[] = {
		{
			{
				0xd1, 0xa3, 0x0b, 0xc2, 0xf9, 0x7a, 0x6e, 0x43,
				0xbf, 0x65, 0xc7, 0xe2, 0x4f, 0xb0, 0xf0, 0xd3
			},	// QMI device service UUID
			1,	// CidCount
			{ 1 }	// QMI CID
		}
	};

	MbimSyncObject_Lock(&syncObject);

	ret = BasicConnectDeviceService_DeviceServiceSubscribeSetSend(
		&transaction,
		pTransport,
		sizeof(elements) / sizeof(elements[0]),
		elements
	);
	if (ret != SUCCESS)
	{
		MbimSyncObject_Unlock(&syncObject);
	}
	else
	{
		ret = MbimSyncObject_TimedWait(&syncObject, 5);
		MbimSyncObject_Unlock(&syncObject);

		if (ret == SUCCESS)
		{
			if (syncObject.status == MBIM_STATUS_SUCCESS)
			{
				MbimEventEntry responseElements[5];
				uint32_t responseElementCount = sizeof(responseElements) / sizeof(responseElements[0]);

				ret = BasicConnectDeviceService_DeviceServiceSubscribeParse(
					syncObject.informationBuffer,
					syncObject.informationBufferLength,
					&responseElementCount,
					responseElements
				);
			}
			else
			{
				ret = FAILURE;
			}

		}
		else
		{
			// TODO: Log timed out or other unexpected error.
			MbimTransport_CancelTransaction(pTransport, transactionId);
			ret = FAILURE;
		}
	}

	MbimSyncObject_Destroy(&syncObject);

	return ret;
}

void IssueMbimReq(const char* pDev)
{
	if (pDev == NULL)
		return;

	MbimTransport transport;
	memset(&transport, 0, sizeof(MbimTransport));

	int ret = MbimTransport_Initialize(&transport, (char*)pDev, 4096);
	dlog(ret == SUCCESS ? eLOG_INFO : eLOG_ERROR, "MbimTransport_Initialize(%s) %s\n", pDev, ret == SUCCESS ? "success" : "failure");
	if (ret == SUCCESS)
	{
		ret = EnableQmiIndications(&transport);
		dlog(ret == SUCCESS ? eLOG_INFO : eLOG_ERROR, "EnableQmiIndications %s\n", ret == SUCCESS ? "success" : "failure");
		sleep(1);

		MbimTransport_ShutDown(&transport);
	}
}

void ShowHelp(const char* appName)
{
    printf("\r\n");
    printf("App usage: \r\n\r\n");
#if QMUX_ROUTER_SUPPORT
	printf("  %s [[[-m|-q|-r] [-d <QMI/MBIM device path> [-u <interface id>]]]|[-R]] [-c <porifle id> [-i ip type]] [-Q|-M <mbim device>] [-tm] [-ts] [-V] [-h]\n\n", appName);
#else
	printf("  %s [-m|-q|-r] [-d <QMI/MBIM device path> [-u <interface id>]] [-c <porifle id> [-i ip type]] [-Q|-M <mbim device>] [-tm] [-ts] [-V] [-h]\n\n", appName);
#endif
	printf("  -d  --device <QMI/MBIM device path>\n");
    printf("        Specifies the QMI or the MBIM device path\n\n");
	printf("  -c  --connect <profile id>\n");
	printf("        Establish data connection on startup, with specified profile id\n\n");
	printf("  -i  --ip <IP type>\n");
	printf("        IP type for automatic data connection on startup. 1: IPV4; 2: IPV6; 3: IPV4V6. Use IPv4v6 if not specified\n\n");
	printf("  -m  --mbim\n");
	printf("        Use MBIM interface (MBIM over PCIe or USB)\n\n");
	printf("  -q  --qmi\n");
	printf("        Use direct QMUX interface (QMI over PCIe, or RmNet over USB)\n\n");
	printf("  -r  --rmnet\n");
	printf("        Use direct QMUX interface (RmNet over USB)\n\n");
	printf("  -u  --usb_interface <interface id>\n");
	printf("        Specifies the USB device interface id (for use on RmNet interface only, ignored if -r option is not specified)\n\n");
	printf("  -Q  --enable_qmap\n");
	printf("        Enable QMAP (not to be used with -M or -r)\n\n");
	printf("  -M  --mbim_device <mbim device>\n");
	printf("        Use MBIM device to enable QMAP (not to be used with -Q)\n\n");
#if QMUX_ROUTER_SUPPORT
	printf("  -R  --router\n");
	printf("        Use QMUX Router\n\n");
#endif
	printf("  -tm  --thermal_mitigation\n");
	printf("        Enable thermal mitigation device monitoring.\n\n");
	printf("  -ts  --thermal_sensor\n" );
    printf("        Enable thermal sensor monitoring.\n\n" );
	printf("  -V  --version\n");
	printf("        Show version information.\n\n");
	printf("  -h  --help\n" );
    printf("        This option prints the usage instructions.\n\n" );
}

// The following function is a callback function for QMUX transport errors.
// It can be used to detect device removal scenarios.
static void TransportErrorCallback(void * pErrCallbackContext, QMUX_TRANSPORT_ERR_INFO err_info)
{
	UNUSEDPARAM(pErrCallbackContext);
	
	dlog(eLOG_INFO, "\nTransport error callback is invoked, error type: %d, errno: %d.\n", err_info.err_type, err_info.errno_val);

	if (err_info.errno_val == ENODEV)
	{
		dlog(eLOG_INFO, "Device has been removed.\n");
	}
}

/*
 * Name:     main
 *
 * Purpose:  Entry point of the application
 *
 * Params:   None
 *
 * Return:   EXIT_SUCCESS, EXIT_FAILURE on unexpected error
 *
 * Notes:    None
 */
int main(int argc, const char *argv[])
{
	printf("\nlite-qmi-connection-manager v%s\n\n", VERSION);

	char dev_path[MAX_PATH_LEN];
	memset(dev_path, 0, MAX_PATH_LEN);

	int nMbimDevIdx = 0;
	bool bHelp = false;
	bool bVer = false;
	bool bRmNet = false;
	bool bEnableQmap = false;
	bool bTmMon = false;
	bool bTsMon = false;
	bool bInvalidArg = false;
	int acProfile = 0;
	int ipType = 3;

	for (int i = 1; i < argc && !bInvalidArg; i++)
	{
		if (strcmp("-d", argv[i]) == 0 || strcmp("--device", argv[i]) == 0)
		{
			if (++i >= argc || argv[i][0] != '/')
				bInvalidArg = true;
			else
				StrCpy(dev_path, argv[i]);
		}
		else if (strcmp("-u", argv[i]) == 0 || strcmp("--usb_interface", argv[i]) == 0)
		{
			if (++i >= argc || argv[i][0] == '-')
				bInvalidArg = true;
			else
				g_rmnet_if_id = atoi(argv[i]);
		}
		else if (strcmp("-c", argv[i]) == 0 || strcmp("--connect", argv[i]) == 0)
			bInvalidArg = ++i >= argc ? true : (acProfile = atoi(argv[i])) == 0;
		else if (strcmp("-i", argv[i]) == 0 || strcmp("--ip", argv[i]) == 0)
		{
			if (++i >= argc || argv[i][0] == '-')
				bInvalidArg = true;
			else
			{
				ipType = atoi(argv[i]);
				bInvalidArg = ipType <= 0 || ipType > 3;
			}
		}
		else if (strcmp("-m", argv[i]) == 0 || strcmp("--mbim", argv[i]) == 0)
		{
			if (g_mode != QMUX_INTERFACE_UNKNOWN)
				bInvalidArg = true;
			g_mode = QMUX_INTERFACE_MBIM;
		}
		else if (strcmp("-q", argv[i]) == 0 || strcmp("--qmi", argv[i]) == 0)
		{
			if (g_mode != QMUX_INTERFACE_UNKNOWN)
				bInvalidArg = true;
			g_mode = QMUX_INTERFACE_DIRECT;
		}
		else if (strcmp("-r", argv[i]) == 0 || strcmp("--rmnet", argv[i]) == 0)
		{
			if (g_mode != QMUX_INTERFACE_UNKNOWN)
				bInvalidArg = true;
			g_mode = QMUX_INTERFACE_DIRECT;
			bRmNet = true;
		}
#if QMUX_ROUTER_SUPPORT
		else if (strcmp("-R", argv[i]) == 0 || strcmp("--router", argv[i]) == 0)
		{
			if (g_mode != QMUX_INTERFACE_UNKNOWN)
				bInvalidArg = true;
			g_mode = QMUX_INTERFACE_ROUTER;
		}
#endif
		else if (strcmp("-M", argv[i]) == 0 || strcmp("--mbim_device", argv[i]) == 0)
		{
			if (++i >= argc || argv[i][0] != '/')
				bInvalidArg = true;
			else
				nMbimDevIdx = i;
		}
		else if (strcmp("-Q", argv[i]) == 0 || strcmp("--enable_qmap", argv[i]) == 0)
			bEnableQmap = true;
		else if (strcmp("-tm", argv[i]) == 0 || strcmp("--thermal_mitigation", argv[i]) == 0)
			bTmMon = true;
		else if (strcmp("-ts", argv[i]) == 0 || strcmp("--thermal_sensor", argv[i]) == 0)
			bTsMon = true;
		else if (strcmp("-np", argv[i]) == 0 || strcmp("--no_ping", argv[i]) == 0)
			g_auto_test_enable = 0;
		else if ((strcmp("-V", argv[i]) == 0 || strcmp("--version", argv[i]) == 0))
		{
			bVer = true;
			if (argc == 2)
				return 0;
		}
		else if (strcmp("-h", argv[i]) == 0 || strcmp("--help", argv[i]) == 0)
			bHelp = true;
		else
			bInvalidArg = true;
	}

	if (bHelp && (argc == 2 || (bVer && argc == 3)))
	{
		ShowHelp(argv[0]);
		return 0;
	}

	if (strlen(dev_path) == 0 || !bRmNet)
		g_rmnet_if_id = RMNET_IF_ID_UNSET;

	if (OpenTransport(&s_Transport, dev_path, sizeof(dev_path), &g_mode, bRmNet ? &g_rmnet_if_id : NULL,
		!bInvalidArg && (nMbimDevIdx == 0 || !bEnableQmap), ShowHelp, argv[0], bHelp) != 0)
		return 0;

	QmuxTransport_RegisterErrCallback(&s_Transport, TransportErrorCallback, NULL);
	
	int mode = QMUX_INTERFACE_UNKNOWN;
	if (g_mode == QMUX_INTERFACE_ROUTER)
		GetDeviceInfo(dev_path, sizeof(dev_path), &mode, NULL, true, NULL, NULL, false);

	if (!GetNetInterfaceName(dev_path, szEthName, 64))
		dlog(eLOG_ERROR, "Unable to detect network interface for %s\n", dev_path);
	else if (CtlService_Initialize(&s_CtlService, &s_Transport) != SUCCESS)
		dlog(eLOG_ERROR, "CtlService_Initialize failed\n");
	else
	{
		memset(&sessions, 0, sizeof(sessions));

		// We use the Ctl service to initialize the regular services because it knows how to 
		// acquire client IDs for them.

		int ret = SUCCESS;

		// Use a do/while for easy exit. We have to clean up.
		do
		{
			if (nMbimDevIdx > 0)
				IssueMbimReq(argv[nMbimDevIdx]);

			// In RmNet mode, Set the raw IP mode by changing /sys/class/net/<adapter name>/qmi/raw_ip to 1 
			// and don't enable qmap here
			if (g_mode == QMUX_INTERFACE_DIRECT && g_rmnet_if_id != RMNET_IF_ID_UNSET)
				EnableQmiMuxRawIp(szEthName);
			else if (bEnableQmap && enable_qmap(true) != SUCCESS)
				dlog(eLOG_ERROR, "enable_qmap failed.\n");

			// Infrastructure is now ready. Let's create some regular QMI services.
			ret = CtlService_InitializeRegularService(&s_CtlService, &s_DmsService, eDMS, DmsIndicationCallback, NULL);
			if (ret != SUCCESS)
			{
				dlog(eLOG_ERROR, "InitializeRegularService eDMS failed.\n");
				break;
			}

			ret = CtlService_InitializeRegularService(&s_CtlService, &s_NasService, eNAS, NasIndicationCallback, NULL);
			if (ret != SUCCESS)
			{
				dlog(eLOG_ERROR, "InitializeRegularService eNAS failed.\n");
				break;
			}

			if (bTmMon)
			{
				ret = CtlService_InitializeRegularService(&s_CtlService, &s_TmdService, eTMD, TmdIndicationCallback, NULL);
				if (ret != SUCCESS)
				{
					dlog(eLOG_ERROR, "InitializeRegularService eTMD failed.\n");
					s_TmdService.serviceType = 0;
				}
			}

			if (bTsMon)
			{
				ret = CtlService_InitializeRegularService(&s_CtlService, &s_TsService, eTS, TsIndicationCallback, NULL);
				if (ret != SUCCESS)
				{
					dlog(eLOG_ERROR, "InitializeRegularService eTS failed.\n");
					s_TsService.serviceType = 0;
				}
			}

			for (int i = 0; i < NUM_WDS_SVC; i++)
			{
				ret = CtlService_InitializeRegularService(&s_CtlService, &sessions[i].wdsSvc, eWDS, WdsIndicationCallback, NULL);
				if (ret != SUCCESS)
				{
					dlog(eLOG_ERROR, "InitializeRegularService eWDS[%d] failed.\n", i);
					break;
				}
			}
			
			ret = CtlService_InitializeRegularService(&s_CtlService, &s_QosService, eQOS, QosIndicationCallback, NULL);
			if (ret != SUCCESS)
			{
				dlog(eLOG_ERROR, "InitializeRegularService eQOS failed.\n");
				s_QosService.serviceType = 0;
			}


		} while (0);

		if (ret == SUCCESS)
			run_tests(acProfile, ipType, bTmMon, bTsMon);

		// Shut down.

		if (g_NasPendingXid != 0)
			QmiService_CancelTransaction(&s_NasService, g_NasPendingXid);

		CtlService_ShutDownRegularService(&s_CtlService, &s_QosService);

		for (int i = 0; i < NUM_WDS_SVC; i++)
			CtlService_ShutDownRegularService(&s_CtlService, &sessions[i].wdsSvc);

		if (bTsMon)
			CtlService_ShutDownRegularService(&s_CtlService, &s_TsService);

		if (bTmMon)
			CtlService_ShutDownRegularService(&s_CtlService, &s_TmdService);

		CtlService_ShutDownRegularService(&s_CtlService, &s_NasService);
		CtlService_ShutDownRegularService(&s_CtlService, &s_DmsService);

		CtlService_ShutDown(&s_CtlService);
	}

	QmuxTransport_RegisterErrCallback(&s_Transport, NULL, NULL);

	QmuxTransport_ShutDown(&s_Transport);

	return 0;
}
